/**
 * The contents of this file are subject to the license and copyright
 * detailed in the LICENSE and NOTICE files at the root of the source
 * tree and available online at
 *
 * http://www.dspace.org/license/
 */
package org.dspace.autoversioning;

import org.dspace.authorize.AuthorizeException;
import org.dspace.authorize.AuthorizeManager;
import org.dspace.authorize.ResourcePolicy;
import org.dspace.content.Bitstream;
import org.dspace.content.Bundle;
import org.dspace.content.DSpaceObject;
import org.dspace.content.Item;
import org.dspace.content.WorkspaceItem;
import org.dspace.core.Context;
import org.dspace.identifier.IdentifierException;
import org.dspace.identifier.IdentifierService;
import org.dspace.utils.DSpace;
import java.io.IOException;
import java.sql.SQLException;
import java.util.List;

import org.dspace.versioning.AbstractVersionProvider;

/**
 *
 *
 * @author Fabio Bolognesi (fabio at atmire dot com)
 * @author Mark Diggory (markd at atmire dot com)
 * @author Ben Bosman (ben at atmire dot com)
 */
public class DefaultItemAutoVersionProvider extends AbstractVersionProvider implements ItemAutoVersionProvider
{

    public Item createNewItemAndAddItInWorkspace(Context context, Item nativeItem) {
        try
        {
            WorkspaceItem workspaceItem = WorkspaceItem.create(context, nativeItem.getOwningCollection(), false);
            Item itemNew = workspaceItem.getItem();
            itemNew.update();
            return itemNew;
        }catch (SQLException e) {
            throw new RuntimeException(e.getMessage(), e);
        }catch (AuthorizeException e) {
            throw new RuntimeException(e.getMessage(), e);
        }catch (IOException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * TODO: CHANGE Previous Item May Not Exist, we need to reinstate using InternalAIPIngester and Archived AIP
     * TODO: Identify where the version gets deleted
     */
    public void deleteVersionedItem(Context c, AutoVersion versionToDelete, AutoVersionHistory history)
    {
        try
        {
            // if versionToDelete is the current version we have to reinstate the previous version
            // and reset canonical

            /** This does not get executed by design right now; latest version is not deletable via UI
            if(history.isLastVersion(versionToDelete) && history.size() > 1)
            {
                // reset the previous version to archived
                Item item = history.getPrevious(versionToDelete).getItem();

             if (item == null) {
             // This was the latest version and a previous version does not exist; this is not permitted
             // TODO: Bring back via InternalAIPIngester...
             } else {
             // This was the latest version and a previous version exists; reinstate the previous version
                item.setArchived(true);
                item.update();
            }
             }
             */

            // assign tombstone to the Identifier and reset canonical to the previous version only if there is a previous version
            IdentifierService identifierService = new DSpace().getSingletonService(IdentifierService.class);
            Item itemToDelete=versionToDelete.getItem();
            if (itemToDelete != null) {
                identifierService.delete(c, itemToDelete);
            }
        } catch (SQLException e) {
            throw new RuntimeException(e.getMessage(), e);
        } catch (AuthorizeException e) {
            throw new RuntimeException(e.getMessage(), e);
        } catch (IdentifierException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * TODO: New Item will keep same handle as previous.
     * TODO: Later We will record version history in AIP Technical Metadata Section
     */
    public Item updateItemState(Context c, Item itemNew, Item previousItem)
    {
        try
        {
            copyMetadata(itemNew, previousItem);
            createBundlesAndAddBitstreams(c, itemNew, previousItem);

            // DSpace knows several types of resource policies (see the class
            // org.dspace.authorize.ResourcePolicy): Submission, Workflow, Custom
            // and inherited. Submission, Workflow and Inherited policies will be
            // set automatically as neccessary. We need to copy the custom policies
            // only to preserve customly set policies and embargos (which are
            // realized by custom policies with a start date).
            List<ResourcePolicy> policies =
                    AuthorizeManager.findPoliciesByDSOAndType(c, previousItem, ResourcePolicy.TYPE_CUSTOM);
            AuthorizeManager.addPolicies(c, policies, itemNew);

            itemNew.update();
            return itemNew;
        }catch (SQLException e) {
            throw new RuntimeException(e.getMessage(), e);
        } catch (AuthorizeException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }
}
