/**
 * The contents of this file are subject to the license and copyright
 * detailed in the LICENSE and NOTICE files at the root of the source
 * tree and available online at
 *
 * http://www.dspace.org/license/
 */
package org.dspace.app.xmlui.aspect.autoversioning;

import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.dspace.app.util.Util;
import org.dspace.app.xmlui.cocoon.AbstractDSpaceTransformer;
import org.dspace.app.xmlui.utils.HandleUtil;
import org.dspace.app.xmlui.utils.UIException;
import org.dspace.app.xmlui.wing.Message;
import org.dspace.app.xmlui.wing.WingException;
import org.dspace.app.xmlui.wing.element.List;
import org.dspace.app.xmlui.wing.element.Options;
import org.dspace.authorize.AuthorizeException;
import org.dspace.authorize.AuthorizeManager;
import org.dspace.autoversioning.AutoVersionHistory;
import org.dspace.content.DSpaceObject;
import org.dspace.content.Item;
import org.dspace.core.Constants;
import org.dspace.utils.DSpace;
import org.dspace.autoversioning.AutoVersioningService;
import org.dspace.versioning.VersioningService;
import org.xml.sax.SAXException;

import java.io.IOException;
import java.sql.SQLException;


/**
 *
 * Navigation for Versioning of Items.
 *
 *
 * @author Fabio Bolognesi (fabio at atmire dot com)
 * @author Mark Diggory (markd at atmire dot com)
 * @author Ben Bosman (ben at atmire dot com)
 */
public class Navigation extends AbstractDSpaceTransformer
{
    private static final Message T_context_head = message("xmlui.administrative.Navigation.context_head");
    private static final Message T_context_create_version= message("xmlui.aspect.versioning.VersioningNavigation.context_create_version");
    private static final Message T_context_show_version_history= message("xmlui.aspect.versioning.VersioningNavigation.context_show_version_history");

    public void addOptions(Options options) throws SAXException, WingException,
            UIException, SQLException, IOException, AuthorizeException {
    	/* Create skeleton menu structure to ensure consistent order between aspects,
    	 * even if they are never used
    	 */
        options.addList("discovery");
        options.addList("browse");
        options.addList("account");

        List content = options.addList("context");


        // Context Administrative options  for Versioning
        DSpaceObject dso = HandleUtil.obtainHandle(objectModel);


        if(dso==null)
        {
            // case: internal-item http://localhost:8100/internal-item?itemID=3085
            // case: admin         http://localhost:8100/admin/item?itemID=3340
            // retrieve the object from the DB
            dso = getItemById();
        }

        if (dso != null && dso.getType() == Constants.ITEM)
        {
            Item item = (Item) dso;
            AutoVersioningService versioningService = (AutoVersioningService) new DSpace().getSingletonService(VersioningService.class);
            boolean canVersion = versioningService.canVersion(this.context,item);
            if(canVersion && item.isArchived())
            {
                boolean headAdded=true;
                content.setHead(T_context_head);
                content.addItem().addXref(contextPath+"/submit?itemID="+item.getID()+"&version=create", T_context_create_version);

                if(hasVersionHistory(item) && item.canEdit())
                {
                    content.addItem().addXref(contextPath+"/item/versionhistory?itemID="+item.getID(), T_context_show_version_history);
                    }
                }
            }
        else if(AuthorizeManager.isAdmin(context)){
            List admin = options.addList("administrative");
            admin.setHead("Administrative");
            List version = admin.addList("versions");
            version.setHead("Version");
            version.addItem().addXref(contextPath+"/admin/versions","Manage Versions");
        }
    }


    private Item getItemById() throws SQLException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        Item item = null;
        int itemId = Util.getIntParameter(request, "itemID");
        if (itemId != -1)
        {
            item = Item.find(this.context, itemId);
        }
        return item;
    }

    private boolean isLatest(Item item)
    {
        AutoVersioningService versioningService = (AutoVersioningService) new DSpace().getSingletonService(VersioningService.class);
        AutoVersionHistory history = versioningService.findAutoVersionHistory(context, item.getID());
        return (history==null || history.getLatestVersion().getItem().getID() == item.getID());
    }


    private boolean hasVersionHistory(Item item)
    {
        AutoVersioningService versioningService = (AutoVersioningService) new DSpace().getSingletonService(VersioningService.class);
        AutoVersionHistory history = versioningService.findAutoVersionHistory(context, item.getID());
        return (history!=null);
    }

}
