/*
 * Decompiled with CFR 0.152.
 */
package org.datatransferproject.transfer;

import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.base.Strings;
import com.google.common.util.concurrent.AbstractScheduledService;
import com.google.inject.Inject;
import java.io.IOException;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import org.datatransferproject.api.launcher.ExtensionContext;
import org.datatransferproject.api.launcher.JobAwareMonitor;
import org.datatransferproject.api.launcher.Monitor;
import org.datatransferproject.launcher.monitor.events.EventCode;
import org.datatransferproject.spi.cloud.storage.JobStore;
import org.datatransferproject.spi.cloud.types.JobAuthorization;
import org.datatransferproject.spi.cloud.types.PortabilityJob;
import org.datatransferproject.spi.transfer.security.PublicKeySerializer;
import org.datatransferproject.spi.transfer.security.TransferKeyGenerator;
import org.datatransferproject.spi.transfer.types.FailureReasons;
import org.datatransferproject.transfer.CredsTimeoutException;
import org.datatransferproject.transfer.JobMetadata;

class JobPollingService
extends AbstractScheduledService {
    private final JobStore store;
    private final TransferKeyGenerator transferKeyGenerator;
    private final PublicKeySerializer publicKeySerializer;
    private final AbstractScheduledService.Scheduler scheduler;
    private final Monitor monitor;
    private final Stopwatch stopwatch = Stopwatch.createUnstarted();
    private final int credsTimeoutSeconds;

    @Inject
    JobPollingService(JobStore store, TransferKeyGenerator transferKeyGenerator, PublicKeySerializer publicKeySerializer, AbstractScheduledService.Scheduler scheduler, Monitor monitor, ExtensionContext context) {
        monitor.debug(() -> "initializing JobPollingService", new Object[0]);
        this.store = store;
        this.transferKeyGenerator = transferKeyGenerator;
        this.publicKeySerializer = publicKeySerializer;
        this.scheduler = scheduler;
        this.monitor = monitor;
        this.credsTimeoutSeconds = (Integer)context.getSetting("credTimeoutSeconds", (Object)300);
        monitor.debug(() -> "initialized JobPollingService", new Object[0]);
    }

    protected void runOneIteration() {
        this.monitor.debug(() -> "JobMetadata.isInitialized(): " + JobMetadata.isInitialized(), new Object[0]);
        if (JobMetadata.isInitialized()) {
            if (this.stopwatch.elapsed(TimeUnit.SECONDS) > (long)this.credsTimeoutSeconds) {
                UUID jobId = JobMetadata.getJobId();
                this.markJobTimedOut(jobId);
                String message = String.format("Waited over %d seconds for the creds to be provided on the claimed job: %s", this.credsTimeoutSeconds, jobId);
                this.monitor.severe(() -> message, new Object[]{EventCode.WORKER_CREDS_TIMEOUT});
                throw new CredsTimeoutException(message, jobId);
            }
            this.pollUntilJobIsReady();
        } else {
            this.pollForUnassignedJob();
        }
    }

    private void markJobTimedOut(UUID jobId) {
        try {
            this.store.markJobAsTimedOut(jobId);
            this.store.addFailureReasonToJob(jobId, FailureReasons.CREDS_TIMEOUT.toString());
        }
        catch (IOException e) {
            this.monitor.severe(() -> String.format("IOException while marking job as timed out. JobId: %s", jobId), new Object[]{e});
        }
    }

    protected AbstractScheduledService.Scheduler scheduler() {
        return this.scheduler;
    }

    private void pollForUnassignedJob() {
        UUID jobId = this.store.findFirst(JobAuthorization.State.CREDS_AVAILABLE);
        this.monitor.debug(() -> "Polling for a job in state CREDS_AVAILABLE", new Object[0]);
        if (jobId == null) {
            this.monitor.debug(() -> "Did not find job after polling", new Object[0]);
            return;
        }
        this.monitor.debug(() -> String.format("Found job %s", jobId), new Object[0]);
        Preconditions.checkState((!JobMetadata.isInitialized() ? 1 : 0) != 0);
        TransferKeyGenerator.WorkerKeyPair keyPair = this.transferKeyGenerator.generate();
        boolean claimed = this.tryToClaimJob(jobId, keyPair);
        if (claimed) {
            this.monitor.debug(() -> String.format("Updated job %s to CREDS_ENCRYPTION_KEY_GENERATED, publicKey length: %s", jobId, keyPair.getEncodedPublicKey().length), new Object[0]);
            this.stopwatch.start();
        }
    }

    private boolean tryToClaimJob(UUID jobId, TransferKeyGenerator.WorkerKeyPair keyPair) {
        PortabilityJob existingJob = this.store.findJob(jobId);
        this.monitor.debug(() -> String.format("JobPollingService: tryToClaimJob: jobId: %s", existingJob), new Object[0]);
        if (existingJob.jobAuthorization().authPublicKey() != null) {
            this.monitor.debug(() -> "A public key cannot be persisted again", new Object[0]);
            return false;
        }
        String scheme = existingJob.jobAuthorization().encryptionScheme();
        if (this.publicKeySerializer == null) {
            this.monitor.severe(() -> String.format("Public key serializer not found for scheme %s processing job: %s", scheme, jobId), new Object[0]);
            return false;
        }
        String serializedKey = this.publicKeySerializer.serialize(keyPair.getEncodedPublicKey());
        PortabilityJob updatedJob = existingJob.toBuilder().setAndValidateJobAuthorization(existingJob.jobAuthorization().toBuilder().setInstanceId(keyPair.getInstanceId()).setAuthPublicKey(serializedKey).setState(JobAuthorization.State.CREDS_ENCRYPTION_KEY_GENERATED).build()).build();
        try {
            this.store.claimJob(jobId, updatedJob);
            this.monitor.debug(() -> String.format("Stored updated job: tryToClaimJob: jobId: %s", existingJob), new Object[0]);
        }
        catch (IOException | IllegalStateException e) {
            this.monitor.debug(() -> String.format("Could not claim job %s. It was probably already claimed by another transfer worker. Error msg: %s", jobId, e.getMessage()), new Object[]{e});
            return false;
        }
        if (this.monitor instanceof JobAwareMonitor) {
            ((JobAwareMonitor)this.monitor).setJobId(jobId.toString());
        }
        JobMetadata.init(jobId, keyPair.getEncodedPrivateKey(), existingJob.transferDataType(), existingJob.exportService(), existingJob.importService(), Stopwatch.createUnstarted());
        this.monitor.debug(() -> String.format("Stored updated job: tryToClaimJob: JobMetadata initialized: %s", jobId), new Object[0]);
        return true;
    }

    private void pollUntilJobIsReady() {
        this.monitor.debug(() -> "pollUntilJobIsReady", new Object[0]);
        UUID jobId = JobMetadata.getJobId();
        PortabilityJob job = this.store.findJob(jobId);
        if (job == null) {
            this.monitor.severe(() -> String.format("Could not poll job %s, it was not present in the key-value store", jobId), new Object[]{EventCode.WORKER_JOB_ERRORED});
            this.stopAsync();
        } else if (job.state() == PortabilityJob.State.CANCELED) {
            this.monitor.info(() -> String.format("Could not poll job %s, it was cancelled", jobId), new Object[]{EventCode.WORKER_JOB_CANCELED});
            this.stopAsync();
        } else if (job.jobAuthorization().state() == JobAuthorization.State.CREDS_STORED) {
            this.monitor.debug(() -> String.format("Polled job %s in state CREDS_STORED", jobId), new Object[0]);
            JobAuthorization jobAuthorization = job.jobAuthorization();
            if (!Strings.isNullOrEmpty((String)jobAuthorization.encryptedAuthData())) {
                this.monitor.debug(() -> String.format("Polled job %s has auth data as expected. Done polling.", jobId), new Object[]{EventCode.WORKER_CREDS_STORED});
            } else {
                this.monitor.severe(() -> String.format("Polled job %s does not have auth data as expected. Done polling this job since it's in a bad state! Starting over.", jobId), new Object[]{EventCode.WORKER_JOB_ERRORED});
            }
            this.stopAsync();
        } else {
            this.monitor.debug(() -> String.format("Polling job %s until it's in state CREDS_STORED. It's currently in state: %s", jobId, job.jobAuthorization().state()), new Object[0]);
        }
    }
}

