/*
 * Decompiled with CFR 0.152.
 */
package org.deepsymmetry.beatlink.dbserver;

import java.io.DataInputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.Socket;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.WritableByteChannel;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;
import org.deepsymmetry.beatlink.CdjStatus;
import org.deepsymmetry.beatlink.Util;
import org.deepsymmetry.beatlink.dbserver.Field;
import org.deepsymmetry.beatlink.dbserver.Message;
import org.deepsymmetry.beatlink.dbserver.NumberField;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Client {
    private static final Logger logger = LoggerFactory.getLogger(Client.class);
    private final Socket socket;
    private final DataInputStream is;
    private final OutputStream os;
    private final WritableByteChannel channel;
    public final int targetPlayer;
    public final int posingAsPlayer;
    public static final NumberField GREETING_FIELD = new NumberField(1L, 4);
    private long transactionCounter = 0L;
    public static final long DEFAULT_MENU_BATCH_SIZE = 64L;
    private static final AtomicLong menuBatchSize = new AtomicLong(64L);
    private final ReentrantLock menuLock = new ReentrantLock();

    Client(Socket socket, int targetPlayer, int posingAsPlayer) throws IOException {
        this.socket = socket;
        this.is = new DataInputStream(socket.getInputStream());
        this.os = socket.getOutputStream();
        this.channel = Channels.newChannel(this.os);
        this.targetPlayer = targetPlayer;
        this.posingAsPlayer = posingAsPlayer;
        try {
            this.sendField(GREETING_FIELD);
            Field response = Field.read(this.is);
            if (!(response instanceof NumberField) || response.getSize() != 4L || ((NumberField)response).getValue() != 1L) {
                throw new IOException("Did not receive expected greeting response from dbserver, instead got: " + response);
            }
            this.performSetupExchange();
        }
        catch (IOException e) {
            this.close();
            throw e;
        }
    }

    private void performSetupExchange() throws IOException {
        Message setupRequest = new Message(0xFFFFFFFEL, Message.KnownType.SETUP_REQ, new NumberField(this.posingAsPlayer, 4));
        this.sendMessage(setupRequest);
        Message response = Message.read(this.is);
        if (response.knownType != Message.KnownType.MENU_AVAILABLE) {
            throw new IOException("Did not receive message type 0x4000 in response to setup message, got: " + response);
        }
        if (response.arguments.size() != 2) {
            throw new IOException("Did not receive two arguments in response to setup message, got: " + response);
        }
        Field player = response.arguments.get(1);
        if (!(player instanceof NumberField)) {
            throw new IOException("Second argument in response to setup message was not a number: " + response);
        }
        if (((NumberField)player).getValue() != (long)this.targetPlayer) {
            throw new IOException("Expected to connect to player " + this.targetPlayer + ", but welcome response identified itself as player " + ((NumberField)player).getValue());
        }
    }

    public boolean isConnected() {
        return this.socket.isConnected();
    }

    void close() {
        try {
            this.channel.close();
        }
        catch (IOException e) {
            logger.warn("Problem closing dbserver client output channel", (Throwable)e);
        }
        try {
            this.os.close();
        }
        catch (IOException e) {
            logger.warn("Problem closing dbserver client output stream", (Throwable)e);
        }
        try {
            this.is.close();
        }
        catch (IOException e) {
            logger.warn("Problem closing dbserver client input stream", (Throwable)e);
        }
        try {
            this.socket.close();
        }
        catch (IOException e) {
            logger.warn("Problem closing dbserver client socket", (Throwable)e);
        }
    }

    private void sendField(Field field) throws IOException {
        if (this.isConnected()) {
            try {
                field.write(this.channel);
            }
            catch (IOException e) {
                logger.warn("Problem trying to write field to dbserver, closing connection", (Throwable)e);
                this.close();
                throw e;
            }
            return;
        }
        throw new IOException("sendField() called after dbserver connection was closed");
    }

    private NumberField assignTransactionNumber() {
        return new NumberField(++this.transactionCounter, 4);
    }

    private void sendMessage(Message message) throws IOException {
        logger.debug("Sending> {}", (Object)message);
        int totalSize = 0;
        for (Field field : message.fields) {
            totalSize += field.getBytes().remaining();
        }
        ByteBuffer combined = ByteBuffer.allocate(totalSize);
        for (Field field : message.fields) {
            logger.debug("..sending> {}", (Object)field);
            combined.put(field.getBytes());
        }
        combined.flip();
        Util.writeFully(combined, this.channel);
    }

    public static NumberField buildRMST(int requestingPlayer, Message.MenuIdentifier targetMenu, CdjStatus.TrackSourceSlot slot) {
        return Client.buildRMST(requestingPlayer, targetMenu, slot, CdjStatus.TrackType.REKORDBOX);
    }

    public static NumberField buildRMST(int requestingPlayer, Message.MenuIdentifier targetMenu, CdjStatus.TrackSourceSlot slot, CdjStatus.TrackType trackType) {
        return new NumberField((requestingPlayer & 0xFF) << 24 | (targetMenu.protocolValue & 0xFF) << 16 | (slot.protocolValue & 0xFF) << 8 | trackType.protocolValue & 0xFF);
    }

    public NumberField buildRMST(Message.MenuIdentifier targetMenu, CdjStatus.TrackSourceSlot slot) {
        return Client.buildRMST(this.posingAsPlayer, targetMenu, slot, CdjStatus.TrackType.REKORDBOX);
    }

    public NumberField buildRMST(Message.MenuIdentifier targetMenu, CdjStatus.TrackSourceSlot slot, CdjStatus.TrackType trackType) {
        return Client.buildRMST(this.posingAsPlayer, targetMenu, slot, trackType);
    }

    public synchronized Message simpleRequest(Message.KnownType requestType, Message.KnownType responseType, Field ... arguments) throws IOException {
        NumberField transaction = this.assignTransactionNumber();
        Message request = new Message(transaction, new NumberField(requestType.protocolValue, 2), arguments);
        this.sendMessage(request);
        Message response = Message.read(this.is);
        if (response.transaction.getValue() != transaction.getValue()) {
            throw new IOException("Received response with wrong transaction ID. Expected: " + transaction.getValue() + ", got: " + response);
        }
        if (responseType != null && response.knownType != responseType) {
            throw new IOException("Received response with wrong type. Expected: " + (Object)((Object)responseType) + ", got: " + response);
        }
        return response;
    }

    public Message menuRequest(Message.KnownType requestType, Message.MenuIdentifier targetMenu, CdjStatus.TrackSourceSlot slot, Field ... arguments) throws IOException {
        return this.menuRequestTyped(requestType, targetMenu, slot, CdjStatus.TrackType.REKORDBOX, arguments);
    }

    public synchronized Message menuRequestTyped(Message.KnownType requestType, Message.MenuIdentifier targetMenu, CdjStatus.TrackSourceSlot slot, CdjStatus.TrackType trackType, Field ... arguments) throws IOException {
        if (!this.menuLock.isHeldByCurrentThread()) {
            throw new IllegalStateException("renderMenuItems() cannot be called without first successfully calling tryLockingForMenuOperation()");
        }
        Field[] combinedArguments = new Field[arguments.length + 1];
        combinedArguments[0] = this.buildRMST(targetMenu, slot, trackType);
        System.arraycopy(arguments, 0, combinedArguments, 1, arguments.length);
        Message response = this.simpleRequest(requestType, Message.KnownType.MENU_AVAILABLE, combinedArguments);
        NumberField reportedRequestType = (NumberField)response.arguments.get(0);
        if (reportedRequestType.getValue() != requestType.protocolValue) {
            throw new IOException("Menu request did not return result for same type as request; sent type: " + requestType.protocolValue + ", received type: " + reportedRequestType.getValue() + ", response: " + response);
        }
        return response;
    }

    public static long getMenuBatchSize() {
        return menuBatchSize.get();
    }

    public static void setMenuBatchSize(long batchSize) {
        menuBatchSize.set(batchSize);
    }

    public boolean tryLockingForMenuOperations(long timeout, TimeUnit unit) throws InterruptedException {
        return this.menuLock.tryLock(timeout, unit);
    }

    public void unlockForMenuOperations() {
        this.menuLock.unlock();
    }

    public synchronized List<Message> renderMenuItems(Message.MenuIdentifier targetMenu, CdjStatus.TrackSourceSlot slot, CdjStatus.TrackType trackType, Message availableResponse) throws IOException {
        long count = availableResponse.getMenuResultsCount();
        if (count == -1L || count == 0L) {
            return Collections.emptyList();
        }
        return this.renderMenuItems(targetMenu, slot, trackType, 0, (int)count);
    }

    public synchronized List<Message> renderMenuItems(Message.MenuIdentifier targetMenu, CdjStatus.TrackSourceSlot slot, CdjStatus.TrackType trackType, int offset, int count) throws IOException {
        if (!this.menuLock.isHeldByCurrentThread()) {
            throw new IllegalStateException("renderMenuItems() cannot be called without first successfully calling tryLockingForMenuOperation()");
        }
        if (offset < 0) {
            throw new IllegalArgumentException("offset must be nonnegative");
        }
        if (count < 1) {
            throw new IllegalArgumentException("count must be positive");
        }
        ArrayList<Message> results = new ArrayList<Message>(count);
        int gathered = 0;
        while (gathered < count) {
            long batchSize = Math.min((long)(count - gathered), menuBatchSize.get());
            NumberField transaction = this.assignTransactionNumber();
            NumberField limit = new NumberField(batchSize);
            NumberField total = new NumberField(count);
            Message request = new Message(transaction, new NumberField(Message.KnownType.RENDER_MENU_REQ.protocolValue, 2), this.buildRMST(targetMenu, slot, trackType), new NumberField(offset), limit, NumberField.WORD_0, total, NumberField.WORD_0);
            this.sendMessage(request);
            Message response = Message.read(this.is);
            if (response.transaction.getValue() != transaction.getValue()) {
                throw new IOException("Received response with wrong transaction ID. Expected: " + transaction.getValue() + ", got: " + response);
            }
            if (response.knownType != Message.KnownType.MENU_HEADER) {
                throw new IOException("Expecting MENU_HEADER, instead got: " + response);
            }
            response = Message.read(this.is);
            while (response.knownType == Message.KnownType.MENU_ITEM) {
                results.add(response);
                response = Message.read(this.is);
            }
            if (response.knownType != Message.KnownType.MENU_FOOTER) {
                throw new IOException("Expecting MENU_ITEM or MENU_FOOTER, instead got: " + response);
            }
            offset = (int)((long)offset + batchSize);
            gathered = (int)((long)gathered + batchSize);
        }
        return Collections.unmodifiableList(results);
    }

    public String toString() {
        return "DBServer Client[targetPlayer: " + this.targetPlayer + ", posingAsPlayer: " + this.posingAsPlayer + ", transactionCounter: " + this.transactionCounter + ", menuBatchSize: " + Client.getMenuBatchSize() + "]";
    }
}

