/*
 * Decompiled with CFR 0.152.
 */
package org.deepsymmetry.beatlink;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.nio.channels.WritableByteChannel;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Util {
    private static final Logger logger = LoggerFactory.getLogger(Util.class);
    private static final byte[] MAGIC_HEADER = new byte[]{81, 115, 112, 116, 49, 87, 109, 74, 79, 76};
    public static final int PACKET_TYPE_OFFSET = 10;
    public static final Map<Integer, Map<Byte, PacketType>> PACKET_TYPE_MAP;
    private static final Map<String, Object> namedLocks;
    private static final Map<String, Integer> namedLockUseCounts;

    public static ByteBuffer getMagicHeader() {
        return ByteBuffer.wrap(MAGIC_HEADER).asReadOnlyBuffer();
    }

    public static DatagramPacket buildPacket(PacketType type, ByteBuffer deviceName, ByteBuffer payload) {
        ByteBuffer content = ByteBuffer.allocate(31 + payload.remaining());
        content.put(Util.getMagicHeader());
        content.put(type.protocolValue);
        content.put(deviceName);
        content.put(payload);
        return new DatagramPacket(content.array(), content.capacity());
    }

    public static PacketType validateHeader(DatagramPacket packet, int port) {
        byte[] data = packet.getData();
        if (data.length < 10) {
            logger.warn("Packet is too short to be a Pro DJ Link packet; must be at least 10 bytes long, was only " + data.length + ".");
            return null;
        }
        if (!Util.getMagicHeader().equals(ByteBuffer.wrap(data, 0, MAGIC_HEADER.length))) {
            logger.warn("Packet did not have correct nine-byte header for the Pro DJ Link protocol.");
            return null;
        }
        Map<Byte, PacketType> portMap = PACKET_TYPE_MAP.get(port);
        if (portMap == null) {
            logger.warn("Do not know any Pro DJ Link packets that are received on port " + port + ".");
            return null;
        }
        PacketType result = portMap.get(data[10]);
        if (result == null) {
            logger.warn("Do not know any Pro DJ Link packets received on port " + port + " with type " + String.format("0x%02x", data[10]) + ".");
        }
        return result;
    }

    public static int unsign(byte b) {
        return b & 0xFF;
    }

    public static long bytesToNumber(byte[] buffer, int start, int length) {
        long result = 0L;
        for (int index = start; index < start + length; ++index) {
            result = (result << 8) + (long)Util.unsign(buffer[index]);
        }
        return result;
    }

    public static long bytesToNumberLittleEndian(byte[] buffer, int start, int length) {
        long result = 0L;
        for (int index = start + length - 1; index >= start; --index) {
            result = (result << 8) + (long)Util.unsign(buffer[index]);
        }
        return result;
    }

    public static void numberToBytes(int number, byte[] buffer, int start, int length) {
        for (int index = start + length - 1; index >= start; --index) {
            buffer[index] = (byte)(number & 0xFF);
            number >>= 8;
        }
    }

    public static long addressToLong(InetAddress address) {
        long result = 0L;
        for (byte element : address.getAddress()) {
            result = (result << 8) + (long)Util.unsign(element);
        }
        return result;
    }

    public static boolean sameNetwork(int prefixLength, InetAddress address1, InetAddress address2) {
        if (logger.isDebugEnabled()) {
            logger.debug("Comparing address " + address1.getHostAddress() + " with " + address2.getHostAddress() + ", prefixLength=" + prefixLength);
        }
        long prefixMask = 0xFFFFFFFFL & (long)(-1 << 32 - prefixLength);
        return (Util.addressToLong(address1) & prefixMask) == (Util.addressToLong(address2) & prefixMask);
    }

    public static double pitchToPercentage(long pitch) {
        return (double)(pitch - 1048567L) / 10485.76;
    }

    public static double pitchToMultiplier(long pitch) {
        return (double)pitch / 1048576.0;
    }

    public static void writeFully(ByteBuffer buffer, WritableByteChannel channel) throws IOException {
        while (buffer.hasRemaining()) {
            channel.write(buffer);
        }
    }

    public static long halfFrameToTime(long halfFrame) {
        return halfFrame * 100L / 15L;
    }

    public static int timeToHalfFrame(long milliseconds) {
        return (int)(milliseconds * 15L / 100L);
    }

    public static synchronized Object allocateNamedLock(String name) {
        Object result = namedLocks.get(name);
        if (result != null) {
            namedLockUseCounts.put(name, namedLockUseCounts.get(name) + 1);
            return result;
        }
        namedLockUseCounts.put(name, 1);
        result = new Object();
        namedLocks.put(name, result);
        return result;
    }

    public static synchronized void freeNamedLock(String name) {
        int count = namedLockUseCounts.get(name);
        if (count > 1) {
            namedLockUseCounts.put(name, count - 1);
        } else {
            namedLocks.remove(name);
            namedLockUseCounts.remove(name);
        }
    }

    private Util() {
    }

    static {
        HashMap scratch = new HashMap();
        for (PacketType packetType : PacketType.values()) {
            HashMap<Byte, PacketType> portMap = (HashMap<Byte, PacketType>)scratch.get(packetType.port);
            if (portMap == null) {
                portMap = new HashMap<Byte, PacketType>();
                scratch.put(packetType.port, portMap);
            }
            portMap.put(packetType.protocolValue, packetType);
        }
        for (Map.Entry entry : scratch.entrySet()) {
            scratch.put((Integer)entry.getKey(), Collections.unmodifiableMap((Map)entry.getValue()));
        }
        PACKET_TYPE_MAP = Collections.unmodifiableMap(scratch);
        namedLocks = new HashMap<String, Object>();
        namedLockUseCounts = new HashMap<String, Integer>();
    }

    public static enum PacketType {
        FADER_START_COMMAND(2, "Fader Start", 50001),
        CHANNELS_ON_AIR(3, "Channels On Air", 50001),
        MEDIA_QUERY(5, "Media Query", 50002),
        MEDIA_RESPONSE(6, "Media Response", 50002),
        DEVICE_KEEP_ALIVE(6, "Device Keep-Alive", 50000),
        CDJ_STATUS(10, "CDJ Status", 50002),
        LOAD_TRACK_COMMAND(25, "Load Track Command", 50002),
        LOAD_TRACK_ACK(26, "Load Track Acknowledgment", 50002),
        MASTER_HANDOFF_REQUEST(38, "Master Handoff Request", 50001),
        MASTER_HANDOFF_RESPONSE(39, "Master Handoff Response", 50001),
        BEAT(40, "Beat", 50001),
        MIXER_STATUS(41, "Mixer Status", 50002),
        SYNC_CONTROL(42, "Sync Control", 50001);

        public final byte protocolValue;
        public final String name;
        public final int port;

        private PacketType(int value, String name, int port) {
            this.protocolValue = (byte)value;
            this.name = name;
            this.port = port;
        }
    }
}

