/*
 * Decompiled with CFR 0.152.
 */
package org.deepsymmetry.beatlink.dbserver;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.ConnectException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketException;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.apiguardian.api.API;
import org.deepsymmetry.beatlink.CdjStatus;
import org.deepsymmetry.beatlink.DeviceAnnouncement;
import org.deepsymmetry.beatlink.DeviceAnnouncementListener;
import org.deepsymmetry.beatlink.DeviceFinder;
import org.deepsymmetry.beatlink.DeviceUpdate;
import org.deepsymmetry.beatlink.LifecycleListener;
import org.deepsymmetry.beatlink.LifecycleParticipant;
import org.deepsymmetry.beatlink.Util;
import org.deepsymmetry.beatlink.VirtualCdj;
import org.deepsymmetry.beatlink.dbserver.Client;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@API(status=API.Status.STABLE)
public class ConnectionManager
extends LifecycleParticipant {
    private static final Logger logger = LoggerFactory.getLogger(ConnectionManager.class);
    private final Map<Integer, Client> openClients = new ConcurrentHashMap<Integer, Client>();
    private final Map<Client, Integer> useCounts = new ConcurrentHashMap<Client, Integer>();
    private final Map<Client, Long> timestamps = new ConcurrentHashMap<Client, Long>();
    private final AtomicInteger idleLimit = new AtomicInteger(1);
    private final Map<InetAddress, Integer> dbServerPorts = new ConcurrentHashMap<InetAddress, Integer>();
    private final DeviceAnnouncementListener announcementListener = new DeviceAnnouncementListener(){

        @Override
        public void deviceFound(DeviceAnnouncement announcement) {
            if (VirtualCdj.getInstance().inOpusQuadCompatibilityMode()) {
                logger.debug("Nothing to do when new devices found in Opus Quad compatibility mode.");
                return;
            }
            if (announcement.getDeviceNumber() == 25 && announcement.getDeviceName().equals("NXS-GW")) {
                logger.debug("Ignoring departure of Kuvo gateway, which fight each other and come and go constantly, especially in CDJ-3000s.");
                return;
            }
            logger.debug("Processing device found, number: {}, name: {}", (Object)announcement.getDeviceNumber(), (Object)announcement.getDeviceName());
            Thread queryThread = new Thread(() -> ConnectionManager.this.requestPlayerDBServerPort(announcement));
            if (ConnectionManager.this.activeQueryThreads.putIfAbsent(announcement.getAddress(), queryThread) == null) {
                queryThread.start();
            }
        }

        @Override
        public void deviceLost(DeviceAnnouncement announcement) {
            if (announcement.getDeviceNumber() == 25 && announcement.getDeviceName().equals("NXS-GW")) {
                logger.debug("Ignoring arrival of Kuvo gateway, which fight each other and come and go constantly, especially in CDJ-3000s.");
                return;
            }
            ConnectionManager.this.dbServerPorts.remove(announcement.getAddress());
        }
    };
    private static final int DB_SERVER_QUERY_PORT = 12523;
    private static final byte[] DB_SERVER_QUERY_PACKET = new byte[]{0, 0, 0, 15, 82, 101, 109, 111, 116, 101, 68, 66, 83, 101, 114, 118, 101, 114, 0};
    private final Map<InetAddress, Thread> activeQueryThreads = new ConcurrentHashMap<InetAddress, Thread>();
    @API(status=API.Status.STABLE)
    public static final int DEFAULT_SOCKET_TIMEOUT = 10000;
    private final AtomicInteger socketTimeout = new AtomicInteger(10000);
    private final AtomicBoolean running = new AtomicBoolean(false);
    private final LifecycleListener lifecycleListener = new LifecycleListener(){

        @Override
        public void started(LifecycleParticipant sender) {
            logger.debug("ConnectionManager does not auto-start when {} starts.", (Object)sender);
        }

        @Override
        public void stopped(LifecycleParticipant sender) {
            if (ConnectionManager.this.isRunning()) {
                logger.info("ConnectionManager stopping because DeviceFinder did.");
                ConnectionManager.this.stop();
            }
        }
    };
    private static final ConnectionManager ourInstance = new ConnectionManager();

    @API(status=API.Status.STABLE)
    public void setIdleLimit(int seconds) {
        if (seconds < 0) {
            throw new IllegalArgumentException("seconds cannot be negative");
        }
        this.idleLimit.set(seconds);
    }

    @API(status=API.Status.STABLE)
    public int getIdleLimit() {
        return this.idleLimit.get();
    }

    private synchronized Client allocateClient(int targetPlayer, String description) throws IOException {
        Client result = this.openClients.get(targetPlayer);
        if (result == null) {
            DeviceAnnouncement targetDeviceAnnouncement = DeviceFinder.getInstance().getLatestAnnouncementFrom(targetPlayer);
            if (targetDeviceAnnouncement == null) {
                throw new IllegalStateException("Player " + targetPlayer + " could not be found " + description);
            }
            int dbServerPort = this.getPlayerDBServerPort(targetPlayer);
            if (dbServerPort < 0) {
                throw new IllegalStateException("Player " + targetPlayer + " does not have a db server " + description);
            }
            byte posingAsPlayerNumber = (byte)this.chooseAskingPlayerNumber(targetDeviceAnnouncement);
            Socket socket = null;
            try {
                InetSocketAddress address = new InetSocketAddress(targetDeviceAnnouncement.getAddress(), dbServerPort);
                socket = new Socket();
                socket.connect(address, this.socketTimeout.get());
                socket.setSoTimeout(this.socketTimeout.get());
                result = new Client(socket, targetPlayer, posingAsPlayerNumber);
            }
            catch (IOException e) {
                try {
                    socket.close();
                }
                catch (IOException e2) {
                    logger.error("Problem closing socket for failed client creation attempt {}", (Object)description);
                }
                throw e;
            }
            this.openClients.put(targetPlayer, result);
            this.useCounts.put(result, 0);
        }
        this.useCounts.put(result, this.useCounts.get(result) + 1);
        return result;
    }

    private void closeClient(Client client) {
        logger.debug("Closing client {}", (Object)client);
        client.close();
        this.openClients.remove(client.targetPlayer);
        this.useCounts.remove(client);
        this.timestamps.remove(client);
    }

    private synchronized void freeClient(Client client) {
        int current = this.useCounts.get(client);
        if (current > 0) {
            this.timestamps.put(client, System.currentTimeMillis());
            this.useCounts.put(client, current - 1);
            if (current == 1 && this.idleLimit.get() == 0) {
                this.closeClient(client);
            }
        } else {
            logger.error("Ignoring attempt to free a client that is not allocated: {}", (Object)client);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @API(status=API.Status.STABLE)
    public <T> T invokeWithClientSession(int targetPlayer, ClientTask<T> task, String description) throws Exception {
        if (!this.isRunning()) {
            throw new IllegalStateException("ConnectionManager is not running, aborting " + description);
        }
        Client client = this.allocateClient(targetPlayer, description);
        try {
            T t = task.useClient(client);
            return t;
        }
        finally {
            this.freeClient(client);
        }
    }

    @API(status=API.Status.STABLE)
    public int getPlayerDBServerPort(int player) {
        this.ensureRunning();
        DeviceAnnouncement announcement = DeviceFinder.getInstance().getLatestAnnouncementFrom(player);
        if (announcement == null) {
            return -1;
        }
        Integer result = this.dbServerPorts.get(announcement.getAddress());
        if (result == null) {
            return -1;
        }
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void requestPlayerDBServerPort(DeviceAnnouncement announcement) {
        try {
            logger.debug("Trying to determine database server port for device {} at IP address {}", (Object)announcement.getDeviceNumber(), (Object)announcement.getAddress().getHostAddress());
            for (int tries = 0; tries < 4; ++tries) {
                if (tries > 0) {
                    try {
                        Thread.sleep(1000 * tries);
                    }
                    catch (InterruptedException e) {
                        logger.info("Interrupted while trying to retry dbserver port query, must be shutting down.");
                        this.activeQueryThreads.remove(announcement.getAddress());
                        return;
                    }
                }
                Socket socket = null;
                try {
                    InetSocketAddress address = new InetSocketAddress(announcement.getAddress(), 12523);
                    socket = new Socket();
                    socket.connect(address, this.socketTimeout.get());
                    InputStream is = socket.getInputStream();
                    OutputStream os = socket.getOutputStream();
                    socket.setSoTimeout(this.socketTimeout.get());
                    os.write(DB_SERVER_QUERY_PACKET);
                    byte[] response = this.readResponseWithExpectedSize(is);
                    if (response.length != 2) continue;
                    int portReturned = (int)Util.bytesToNumber(response, 0, 2);
                    if (logger.isInfoEnabled()) {
                        String suffix = portReturned == 65535 ? ", not yet ready?" : ".";
                        logger.info("Device {} at address {} reported dbserver port of {}{}", new Object[]{announcement.getDeviceNumber(), announcement.getAddress().getHostAddress(), portReturned, suffix});
                    }
                    if (this.isRunning()) {
                        this.dbServerPorts.put(announcement.getAddress(), portReturned);
                    }
                    return;
                }
                catch (ConnectException ce) {
                    logger.info("Device {} at address {} doesn't answer rekordbox port queries, connection refused, not yet ready?", (Object)announcement.getDeviceNumber(), (Object)announcement.getAddress().getHostAddress());
                    continue;
                }
                catch (Throwable t) {
                    logger.warn("Problem requesting database server port number", t);
                    continue;
                }
                finally {
                    if (socket != null) {
                        try {
                            socket.close();
                        }
                        catch (IOException e) {
                            logger.warn("Problem closing database server port request socket", (Throwable)e);
                        }
                    }
                }
            }
            logger.info("Device {} at address {} never responded with a valid rekordbox dbserver port. Won't attempt to request metadata.", (Object)announcement.getDeviceNumber(), (Object)announcement.getAddress().getHostAddress());
        }
        catch (Throwable t) {
            logger.error("Problem querying for database server port on device {} at address {}:", new Object[]{announcement.getDeviceNumber(), announcement.getAddress().getHostAddress(), t});
        }
        finally {
            this.activeQueryThreads.remove(announcement.getAddress());
        }
    }

    @API(status=API.Status.STABLE)
    public void setSocketTimeout(int timeout) {
        this.socketTimeout.set(timeout);
    }

    @API(status=API.Status.STABLE)
    public int getSocketTimeout() {
        return this.socketTimeout.get();
    }

    private byte[] receiveBytes(InputStream is) throws IOException {
        byte[] buffer = new byte[8192];
        int len = is.read(buffer);
        if (len < 1) {
            throw new IOException("receiveBytes read " + len + " bytes.");
        }
        return Arrays.copyOf(buffer, len);
    }

    private byte[] readResponseWithExpectedSize(InputStream is) throws IOException {
        byte[] result = this.receiveBytes(is);
        if (result.length != 2) {
            logger.warn("Expected 2 bytes while reading database server port query packet response, received {}", (Object)result.length);
        }
        return result;
    }

    private int chooseAskingPlayerNumber(DeviceAnnouncement targetPlayer) {
        byte fakeDevice = VirtualCdj.getInstance().getDeviceNumber();
        if (fakeDevice > 4 && !DeviceFinder.getInstance().isDeviceMetadataLimited(targetPlayer)) {
            return targetPlayer.getDeviceNumber();
        }
        if (targetPlayer.getDeviceNumber() > 15 || fakeDevice >= 1 && fakeDevice <= 4) {
            return fakeDevice;
        }
        for (DeviceAnnouncement candidate : DeviceFinder.getInstance().getCurrentDevices()) {
            DeviceUpdate lastUpdate;
            int realDevice = candidate.getDeviceNumber();
            if (realDevice == targetPlayer.getDeviceNumber() || realDevice < 1 || realDevice > 4 || !((lastUpdate = VirtualCdj.getInstance().getLatestStatusFor(realDevice)) instanceof CdjStatus) || ((CdjStatus)lastUpdate).getTrackSourcePlayer() == targetPlayer.getDeviceNumber()) continue;
            return candidate.getDeviceNumber();
        }
        throw new IllegalStateException("No player number available to query player " + targetPlayer + ". If such a player is present on the network, it must be using Link to play a track from our target player, so we can't steal its channel number.");
    }

    @Override
    @API(status=API.Status.STABLE)
    public boolean isRunning() {
        return this.running.get();
    }

    private synchronized void closeIdleClients() {
        LinkedList<Client> candidates = new LinkedList<Client>(this.openClients.values());
        logger.debug("Scanning for idle clients; {} candidates.", (Object)candidates.size());
        for (Client client : candidates) {
            if (this.useCounts.get(client) >= 1 || this.timestamps.get(client) + (long)this.idleLimit.get() * 1000L > System.currentTimeMillis()) continue;
            logger.debug("Idle time reached for unused client {}", (Object)client);
            this.closeClient(client);
        }
    }

    @API(status=API.Status.STABLE)
    public synchronized void start() throws SocketException {
        if (!this.isRunning()) {
            DeviceFinder.getInstance().addLifecycleListener(this.lifecycleListener);
            DeviceFinder.getInstance().addDeviceAnnouncementListener(this.announcementListener);
            DeviceFinder.getInstance().start();
            for (DeviceAnnouncement device : DeviceFinder.getInstance().getCurrentDevices()) {
                this.announcementListener.deviceFound(device);
            }
            new Thread(null, () -> {
                while (this.isRunning()) {
                    try {
                        Thread.sleep(500L);
                    }
                    catch (InterruptedException e) {
                        logger.warn("Interrupted sleeping to close idle dbserver clients");
                    }
                    this.closeIdleClients();
                }
                logger.info("Idle dbserver client closer shutting down.");
            }, "Idle dbserver client closer").start();
            this.running.set(true);
            this.deliverLifecycleAnnouncement(logger, true);
        }
    }

    @API(status=API.Status.STABLE)
    public synchronized void stop() {
        if (this.isRunning()) {
            this.running.set(false);
            DeviceFinder.getInstance().removeDeviceAnnouncementListener(this.announcementListener);
            for (Thread thread : this.activeQueryThreads.values()) {
                thread.interrupt();
            }
            this.dbServerPorts.clear();
            for (Client client : this.openClients.values()) {
                try {
                    client.close();
                }
                catch (Exception e) {
                    logger.warn("Problem closing {} when stopping", (Object)client, (Object)e);
                }
            }
            this.openClients.clear();
            this.useCounts.clear();
            this.deliverLifecycleAnnouncement(logger, false);
        }
    }

    @API(status=API.Status.STABLE)
    public static ConnectionManager getInstance() {
        return ourInstance;
    }

    private ConnectionManager() {
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("ConnectionManager[running:").append(this.isRunning());
        sb.append(", dbServerPorts:").append(this.dbServerPorts).append(", openClients:").append(this.openClients);
        sb.append(", useCounts:").append(this.useCounts).append(", timestamps:").append(this.timestamps);
        return sb.append(", idleLimit:").append(this.idleLimit.get()).append("]").toString();
    }

    @API(status=API.Status.STABLE)
    public static interface ClientTask<T> {
        public T useClient(Client var1) throws Exception;
    }
}

