/*
 * @Copyright 2010, MechSoft 
 * MechSoft, Mechanical and Software Solutions 
 * 
 * Licensed under the Apache License, Version 2.0 (the 
 * "License"); you may not use this file except in compliance 
 * with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.defne.ejb.service;


import java.util.Set;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.ejb.EJBContext;
import javax.ejb.TransactionAttribute;
import javax.ejb.TransactionAttributeType;
import javax.interceptor.ExcludeClassInterceptors;
import javax.interceptor.Interceptors;

import org.defne.service.Message;
import org.defne.service.annotation.ServiceScanner;
import org.defne.service.annotation.ServiceScanner.ServiceMethod;
import org.defne.service.exception.ServiceNotFoundException;
import org.defne.service.exception.SecurityException;
import org.defne.service.executor.IServiceExecutor;
import org.defne.utility.Utility;
import org.defne.utility.exception.DefneException;
import org.defne.utility.reflection.ReflectionUtils;

/**
 * Imlementation of the {@link IServiceExecutor}.
 * <p>
 * This stateless ejb is responsible for executing the service method.
 * </p>
 * 
 * @author gurkanerdogdu
 * @version $Rev$ $Date$
 */
@Interceptors(value={TransactionalInterceptor.class})
public class ServiceExecutor implements IServiceExecutor
{
    /**Service scanner instance*/
    private ServiceScanner serviceScanner = null;
    
    /**Environment context of the session bean*/
    private @Resource EJBContext context;
    
    /**
     * Default constructor.
     */
    public ServiceExecutor()
    {
        
    }
            
    /**
     * Execute with transaction.
     * 
     * @param inputBag the input bag
     * @return the service bag
     * @throws DefneException the green exception {@inheritDoc}
     */
    @Override
    @TransactionAttribute(TransactionAttributeType.REQUIRED)
    public Message executeWithTransaction(Message inputBag) throws DefneException
    {
        return execute(inputBag);
    }

    /**
     * Execute with new transaction.
     * 
     * @param inputBag the input bag
     * @return the service bag
     * @throws DefneException the green exception {@inheritDoc}
     */
    @Override
    @TransactionAttribute(TransactionAttributeType.REQUIRES_NEW)
    public Message executeWithNewTransaction(Message inputBag) throws DefneException
    {
        return execute(inputBag);
    }

    /**
     * Execute without transaction.
     * 
     * @param inputBag the input bag
     * @return the service bag
     * @throws DefneException the green exception {@inheritDoc}
     */
    @Override
    @TransactionAttribute(TransactionAttributeType.NEVER)
    @ExcludeClassInterceptors
    public Message executeWithoutTransaction(Message inputBag) throws DefneException
    {
        return execute(inputBag);
    }
    
    /**
     * Setup the service scanner.
     */
    @PostConstruct
    protected void postConstruct()
    {
        this.serviceScanner = ServiceScanner.getInstance();
    }
    
    /**
     * Execute the given service.
     * 
     * @param inputBag the input bag
     * @return the service bag
     * @throws DefneException the exception
     */
    private Message execute(Message inputBag) throws DefneException
    {
        Message outputBag = null;
        try
        {
            ServiceMethod serviceMethod = this.serviceScanner.getServiceMethod(ReflectionUtils.getThreadClassLoader(), 
                    inputBag.getServiceName(), inputBag.getServiceMethodName());
            
            if(serviceMethod == null)
            {
                throw new ServiceNotFoundException("Service with name : " + inputBag.getServiceName() + ", method : " +
                        inputBag.getServiceMethodName() + " not found");
            }            
            
            Set<String> roles = serviceMethod.getRoles();
            boolean securityPass = false;
            for(String role : roles)
            {
                if(this.context.isCallerInRole(role))
                {
                    securityPass = true;
                    break;
                }
            }
            
            if(!securityPass)
            {
                throw new SecurityException("You are not authorized to execute service method. Service name : " + 
                        inputBag.getServiceName() + ", service method : " + inputBag.getServiceMethodName());
            }            

            outputBag = (Message) serviceMethod.getServiceMethod().invoke(null, new Object[]{inputBag});
        }
        catch (Exception e)
        {
            //Throws as defne exception, 
            //it rollbacks the transaction
            Utility.throwsDefneException(e);
        }
        return outputBag;
    }

}
