/*
 * @Copyright 2010, MechSoft 
 * MechSoft, Mechanical and Software Solutions 
 * 
 * Licensed under the Apache License, Version 2.0 (the 
 * "License"); you may not use this file except in compliance 
 * with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.defne.ejb.service;

import javax.annotation.PostConstruct;
import javax.interceptor.AroundInvoke;
import javax.interceptor.InvocationContext;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;

import org.defne.jpa.impl.ServiceBagEnitityManagerProvider;
import org.defne.service.IMessageConstants;
import org.defne.service.Message;
import org.defne.utility.exception.DefneException;
import org.defne.utility.log.ILogger;
import org.defne.utility.log.LoggerProvider;

/**
 * EJB based interceptor. It is responsible for
 * @author gurkanerdogdu
 * @version $Rev$ $Date$
 */
public class TransactionalInterceptor
{
    //Logger instance
    private static final ILogger logger = LoggerProvider.getLogProvider(TransactionalInterceptor.class);
    
    /**Injection of the default named entity manager
     * <p>
     * Currently applications must provide the "default"
     * unit name persistence.xml.
     * </p> 
     */ 
    private @PersistenceContext(unitName="default") EntityManager entityManager;
    
    /**Entity manager provider instance*/
    private ServiceBagEnitityManagerProvider entityManagerProvider;
    
    /**
     * Setups entity manager provider instance
     * that holds the actual entity manager.
     * @param ctx invocation context
     */
    @PostConstruct
    public void postConstruct(InvocationContext ctx)
    {
        this.entityManagerProvider = new ServiceBagEnitityManagerProvider(this.entityManager);
    }
    
    /**
     * Intercepts the call and inject entity manager to bag
     * that is used by the service methods.
     * @param context invocation context
     * @return the service execution result
     * @throws Exception for any exception
     */
    @AroundInvoke
    public Object aroundServiceMethodInvoke(InvocationContext context) throws Exception
    {
        Object result = null;
        Message param = null;
        try
        {
             Object[] params = context.getParameters();
             if(params == null || params.length != 1 || params[0].getClass() != Message.class)
             {
                 logger.error("Method parameter must be Message, unknown method in " + context.getMethod());
                 throw new DefneException("Method parameter must be Message, unknown method in " + context.getMethod());
             }
             
             param = Message.class.cast(params[0]);
             param.putMessageParameter(IMessageConstants.SERVICE_CALLER_ENTITY_MANAGER_KEY, this.entityManagerProvider);
                
             //Call actual EJB method
             result = context.proceed();
            
        }finally
        {
            //Remove EM from the bag
            param.removeMessageParameter(IMessageConstants.SERVICE_CALLER_ENTITY_MANAGER_KEY);
        }
        
        return result;
    }
 
}
