/*
 * @Copyright 2010, MechSoft 
 * MechSoft, Mechanical and Software Solutions 
 * 
 * Licensed under the Apache License, Version 2.0 (the 
 * "License"); you may not use this file except in compliance 
 * with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.defne.owb;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.enterprise.context.ApplicationScoped;
import javax.inject.Named;

import org.defne.owb.bindings.EjbExecutorProxy;
import org.defne.service.IMessageConstants;
import org.defne.service.Message;
import org.defne.service.MessageFactory;
import org.defne.service.executor.IServiceExecutorProxy;
import org.defne.service.executor.ServiceExecutorProxyProvider;
import org.defne.service.executor.ServiceExecutorProxyTypes;
import org.defne.service.util.ServiceUtility;
import org.defne.utility.exception.DefneException;
import org.defne.utility.log.ILogger;
import org.defne.utility.log.LoggerProvider;

/**
 * JSR-299 based bean that is reposible for
 * executing the EJB service method executor.
 * 
 * <p>
 * This bean's scope is {@link ApplicationScoped}, therefore
 * it lived along the application lifecycle.
 * </p>
 * @author gurkanerdogdu
 * @version $Rev$ $Date$
 */
@Named("OwbEjbServiceExecutorProxy")
@ApplicationScoped
@EjbExecutorProxy
public class OwbEjbServiceExecutorProxy implements IServiceExecutorProxy
{
    /**Logger instance*/
    private  static final ILogger logger = LoggerProvider.getLogProvider(OwbEjbServiceExecutorProxy.class);
    
    /**Ejb JNDI name*/
    private String ejbJndiName;
    
    /**Actual proxy, that is EJB service method executor proxy*/
    private IServiceExecutorProxy proxy;
    
    /**
     * Createa an instance.
     */
    public OwbEjbServiceExecutorProxy()
    {
        //No operation
    }
    
    /**
     * Creating a proxy instance.
     */
    @PostConstruct
    public void init()
    {
        //Ejb service method executor
        this.proxy = ServiceExecutorProxyProvider.getServiceExecutorProxy(ServiceExecutorProxyTypes.EJB);        
    }
    
    /**
     * {@inheritDoc}
     */
    public Message execute(Message inputBag)
    {
        Message outputBag = MessageFactory.newMessage(inputBag.getServiceName(), inputBag.getServiceMethodName());
        try
        {
            if(this.ejbJndiName == null)
            {
                throw new DefneException("Service Executor EJB Jndi name is null");
            }
            
            inputBag.putMessageParameter(IMessageConstants.SERVICE_EXECUTOR_EJB_JNDI_NAME_KEY, this.ejbJndiName);
            outputBag = this.proxy.execute(inputBag);
        }
        catch (DefneException exception)
        {
            if(logger.isErrorEnabled())
            {
                logger.error(ServiceUtility.getErrorMessage(outputBag,exception),exception.getCause());                
                outputBag = ServiceUtility.getServiceExecutionError(outputBag, exception);
            }
        }
        return outputBag;
    }    

    /**
     * @return the ejbJndiName
     */
    public String getEjbJndiName()
    {
        return ejbJndiName;
    }

    /**
     * @param ejbJndiName the ejbJndiName to set
     */
    public void setEjbJndiName(String ejbJndiName)
    {
        this.ejbJndiName = ejbJndiName;
    }
    
    /**
     * Predestroy actions.
     */
    @PreDestroy
    public void destroy()
    {
        clear(this.ejbJndiName);
    }
    
    /**
     * {@inheritDoc}
     */
    @Override
    public void clear(Object clearObject)
    {
        this.proxy.clear(clearObject);        
    }
}
