/*
 * @Copyright 2010, MechSoft 
 * MechSoft, Mechanical and Software Solutions 
 * 
 * Licensed under the Apache License, Version 2.0 (the 
 * "License"); you may not use this file except in compliance 
 * with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.defne.owb;

import java.util.Set;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.enterprise.context.ApplicationScoped;
import javax.inject.Inject;
import javax.inject.Named;
import javax.persistence.EntityManagerFactory;

import org.defne.jpa.api.EmfProvider;
import org.defne.jpa.impl.ServiceBagEnitityManagerProvider;
import org.defne.owb.bindings.PojoExecutorProxy;
import org.defne.security.spi.IPrincipal;
import org.defne.service.IMessageConstants;
import org.defne.service.Message;
import org.defne.service.MessageFactory;
import org.defne.service.TransactionPolicy;
import org.defne.service.annotation.ServiceScanner;
import org.defne.service.annotation.ServiceScanner.ServiceMethod;
import org.defne.service.exception.ServiceNotFoundException;
import org.defne.service.exception.SecurityException;
import org.defne.service.executor.IPojoServiceExecutor;
import org.defne.service.executor.IServiceExecutorProxy;
import org.defne.service.util.ServiceUtility;
import org.defne.utility.exception.DefneException;
import org.defne.utility.factories.SingletonProvider;
import org.defne.utility.log.ILogger;
import org.defne.utility.log.LoggerProvider;
import org.defne.utility.reflection.ReflectionUtils;

@Named("OwbPojoServiceExecutorProxy")
@ApplicationScoped
@PojoExecutorProxy
public class OwbPojoServiceExecutorProxy implements IServiceExecutorProxy
{
    private static final ILogger logger = LoggerProvider.getLogProvider(OwbPojoServiceExecutorProxy.class);
    
    /**Pojo Beans service executor*/
    private @Inject IPojoServiceExecutor pojoServiceExecutor;
    
    /**Request Scoped Principal support*/
    private @Inject  IPrincipal principal;
    
    /**Application scanner instance*/
    private ServiceScanner serviceScanner = null;
    
    /**
     * Creates a new instance.
     */
    public OwbPojoServiceExecutorProxy()
    {
        
    }
        
    /**
     * {@inheritDoc}
     */
    @Override
    public void clear(Object clearObject)
    {
        this.pojoServiceExecutor = null;
    }
    
    @PostConstruct
    protected void postConstruct()
    {
        this.serviceScanner = ServiceScanner.getInstance();
    }
    
    /**
     * {@inheritDoc}
     */
    @Override
    public Message execute(Message inBag) throws DefneException
    {
        Message oBag = MessageFactory.newMessage(inBag.getServiceName(),inBag.getServiceMethodName());        
        try
        {
            ServiceMethod serviceMethod = this.serviceScanner.getServiceMethod(ReflectionUtils.getThreadClassLoader(), 
                    inBag.getServiceName(), inBag.getServiceMethodName());

            if(serviceMethod == null)
            {
                throw new ServiceNotFoundException("Service with name : " + inBag.getServiceName() + ", method : " +
                        inBag.getServiceMethodName() + " not found");
            }            
            
            String[] userRoles = this.principal.roles();
            Set<String> serviceRoles = serviceMethod.getRoles();
            boolean securityPass = false;
            if(serviceRoles.isEmpty())
            {
                securityPass = true;
            }
            else
            {
                for(String serviceRole : serviceRoles)
                {
                    if(userRoles != null && userRoles.length > 0)
                    {
                        for(String userRole : userRoles)
                        {
                            if(userRole.equals(serviceRole))
                            {
                                securityPass = true;
                                break;
                            }
                        }
                        
                        if(securityPass)
                        {
                            break;
                        }
                    }
                }
            }
            
            if(!securityPass)
            {
                throw new SecurityException("You are not authorized to execute service method. Service name : " + 
                        inBag.getServiceName() + ", service method : " + inBag.getServiceMethodName());
            }
            
            inBag.putMessageParameter("METHOD", serviceMethod.getServiceMethod());
            
            //Checking transaction level
            TransactionPolicy transactionPolicy = null;            
            if(inBag.isExist(IMessageConstants.SERVICE_CALLER_TRANSACTION_POLICY))
            {
                transactionPolicy = inBag.getMessageParameter(TransactionPolicy.class, IMessageConstants.SERVICE_CALLER_TRANSACTION_POLICY);

            }            
            else
            {
                //Lets check with annotations
                transactionPolicy = serviceMethod.getTransactionPolicy();
            }
            
            
            boolean useJpaLogic = false;
            
            //Override JPA Logic with Key
            if(inBag.isExist(IMessageConstants.SERVICE_CALLER_USES_ENTITY_MANAGER_KEY))
            {
                useJpaLogic = inBag.getMessageParameter(Boolean.class, IMessageConstants.SERVICE_CALLER_USES_ENTITY_MANAGER_KEY);                
            }
            else if(transactionPolicy.equals(TransactionPolicy.WITH_TRANSACTION) ||
                    transactionPolicy.equals(TransactionPolicy.NEW_TRANSACTION))
            {
                useJpaLogic = true;
            }            
            else
            {
                useJpaLogic = serviceMethod.isUseEm();
            }
            
            //Service uses @EntityManager or not
            //If uses, use old or creates new
            if(useJpaLogic)
            {
                String emfUnitName = null;
                
                //EMF unit name is overriden by bag key
                if(inBag.isExist(IMessageConstants.SERVICE_JPA_EMF_UNIT_NAME_KEY))
                {
                    emfUnitName = inBag.getMessageParameter(String.class, IMessageConstants.SERVICE_JPA_EMF_UNIT_NAME_KEY);
                }
                else
                {
                    emfUnitName = serviceMethod.getEmUnitName();
                }
                
                //Gets provider
                EmfProvider emfProvider = (EmfProvider) SingletonProvider.getSingletonInstance(SingletonProvider.DEFAULT_EMF_PROVIDER_CLASS, 
                            ReflectionUtils.getThreadClassLoader());
                
                EntityManagerFactory factory = emfProvider.getEntityManagerFactory(emfUnitName);
                if(factory == null)
                {
                    throw new DefneException("There is no JPA EntityManagerFactory for the EMF unit name : " + emfUnitName);
                }
                
                boolean putEntityManager = false;
                if(inBag.isExist(IMessageConstants.SERVICE_CALLER_ENTITY_MANAGER_KEY))
                {
                    if(inBag.isExist(IMessageConstants.SERVICE_CALLER_NEW_ENTITY_MANAGER_KEY))
                    {
                        putEntityManager = inBag.getMessageParameter(Boolean.class, 
                                IMessageConstants.SERVICE_CALLER_NEW_ENTITY_MANAGER_KEY);
                    }
                    else
                    {
                        putEntityManager = transactionPolicy.equals(TransactionPolicy.NEW_TRANSACTION);
                    }                                        
                }
                else
                {
                    putEntityManager = true;
                }

                //Put New EntityManager or not
                if(putEntityManager)
                {
                    inBag.putMessageParameter(IMessageConstants.SERVICE_CALLER_ENTITY_MANAGER_KEY, 
                            new ServiceBagEnitityManagerProvider(factory.createEntityManager()));                                        
                }                
            }
            
            //No transaction
            if(transactionPolicy.equals(TransactionPolicy.NO_TRANSACTION))
            {
                oBag = this.pojoServiceExecutor.executeWithoutTransaction(inBag);
            }
            
            //With transaction
            else
            {
                oBag = this.pojoServiceExecutor.executeWithTransaction(inBag);
            }
            
        }catch(DefneException e)
        {
            if(logger.isErrorEnabled())
            {
                logger.error(ServiceUtility.getErrorMessage(inBag, e),e.getCause());                
                oBag = ServiceUtility.getServiceExecutionError(oBag, e);
            }
        }
        
        
        return oBag;
    }
    
    
    @PreDestroy
    protected void preDestroy()
    {
        EmfProvider emfProvider = (EmfProvider) SingletonProvider.getSingletonInstance(SingletonProvider.DEFAULT_EMF_PROVIDER_CLASS, 
                ReflectionUtils.getThreadClassLoader());
        
        emfProvider.removeEntityManagerFactories();
        
        this.pojoServiceExecutor = null;
        this.principal = null;
        this.serviceScanner = null;
    }

}
