/*
 * @Copyright 2010, MechSoft 
 * MechSoft, Mechanical and Software Solutions 
 * 
 * Licensed under the Apache License, Version 2.0 (the 
 * "License"); you may not use this file except in compliance 
 * with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.defne.owb.interceptors;

import javax.interceptor.AroundInvoke;
import javax.interceptor.Interceptor;
import javax.interceptor.InvocationContext;
import javax.persistence.EntityManager;
import javax.persistence.EntityTransaction;

import org.defne.jpa.impl.ServiceBagEnitityManagerProvider;
import org.defne.owb.interceptors.binding.Transactional;
import org.defne.service.IMessageConstants;
import org.defne.service.Message;
import org.defne.utility.exception.DefneException;
import org.defne.utility.log.ILogger;
import org.defne.utility.log.LoggerProvider;

/**
 * Transactional interceptor for handling
 * the transactional logic around business method calls.
 * @author gurkanerdogdu
 * @version $Id$
 */
@Interceptor
@Transactional
public class TransactionalInterceptor
{
    private static final ILogger logger = LoggerProvider.getLogProvider(TransactionalInterceptor.class);
    
    /**
     * Pre-method called actions.
     * @param invocationContext invocation context
     * @return the result
     * @throws Exception for exception
     */
    @AroundInvoke
    public Object aroundInvoke(InvocationContext invocationContext) throws Exception
    {
        if(logger.isTraceEnabled())
        {
            logger.trace("Calling method : " + invocationContext.getMethod());
        }
        
        boolean active = false;    
        EntityManager manager = null;
        EntityTransaction entityTransaction = null;
        ServiceBagEnitityManagerProvider provider = null;
        Message inputBag = null;
        try
        {
             inputBag = (Message) invocationContext.getParameters()[0];
             if(!inputBag.isExist(IMessageConstants.SERVICE_CALLER_ENTITY_MANAGER_KEY))
             {
                 throw new DefneException("EntityManager is null but it is required to execute service name : " + 
                         inputBag.getServiceName() + " for service method : " + inputBag.getServiceMethodName());
             }
             
              provider = inputBag.getMessageParameter(ServiceBagEnitityManagerProvider.class, 
                     IMessageConstants.SERVICE_CALLER_ENTITY_MANAGER_KEY);
             
             if(provider == null || ((manager = provider.getEntityManager()) == null))
             {
                 throw new DefneException("EntityManager is null but it is required to execute service name : " + 
                         inputBag.getServiceName() + " for service method : " + inputBag.getServiceMethodName());                 
             }
             
             entityTransaction = manager.getTransaction();
             
             if(entityTransaction != null)
             {
                 active = entityTransaction.isActive();    
             }
             
             if(!active)
             {
                 entityTransaction.begin();
             }             
             
             return invocationContext.proceed();
            
        }catch(Exception e)
        {
            if(!active)
            {
                if(entityTransaction != null && entityTransaction.isActive())
                {
                    entityTransaction.rollback();
                }                
            }
            
            throw e;
            
        }finally
        {
            if(!active)
            {
                if(entityTransaction != null && entityTransaction.isActive())
                {
                    entityTransaction.commit();
                }            
                
                if(manager != null)
                {
                    manager.close();
                }     
                
                if(provider != null)
                {
                    provider.destroyEntityManager();
                    inputBag.removeMessageParameter(IMessageConstants.SERVICE_CALLER_ENTITY_MANAGER_KEY);
                }
            }
        }        
    }
}
