/*
 * @Copyright 2010, MechSoft 
 * MechSoft, Mechanical and Software Solutions 
 * 
 * Licensed under the Apache License, Version 2.0 (the 
 * "License"); you may not use this file except in compliance 
 * with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.defne.utility;

import java.lang.reflect.InvocationTargetException;

import org.defne.utility.exception.DefneException;

/**
 * Some utility methods for using all
 * over the project.
 * 
 * @version $Rev: 55 $ $Date: 2010-07-06 14:36:36 +0300 (Tue, 06 Jul 2010) $
 * 
 */
public final class Utility
{
    /**
     * Private constructor.
     */
    private Utility()
    {
        throw new UnsupportedOperationException();
    }
    
    /**
     * Gets target exception if the given exception
     * is {@link InvocationTargetException}.
     * @param exception throwed exception
     * @return the target exception 
     */
    public static Throwable getTargetException(Exception exception)
    {
        Assertions.checkForNull(exception, "exception parameter is null");
        if(exception instanceof InvocationTargetException)
        {
            Throwable target = ((InvocationTargetException)exception).getTargetException();
            if(target != null)
            {
                return target;
            }
        }
        
        return exception;
    }
    
    /**
     * Check and throws defne exception.
     * @param throwable cause
     */
    public static void throwsDefneException(Exception throwable)
    {
        Assertions.checkForNull(throwable, "throwable parameter is null");
        if(throwable instanceof DefneException)
        {
            throw (DefneException)throwable;
        }
        else
        {
            Throwable e = getTargetException(throwable);
            if(e instanceof DefneException)
            {
                throw (DefneException)e;
            }
            
            throw new DefneException(e);
        }
    }
    
    /**
     * Gets system property value with the given name.
     * @param propertyName system property name
     * @return system property value
     */
    public static String getSystemProperty(String propertyName)
    {
        Assertions.checkForNullParameter(propertyName, "propertyName");
        return System.getProperty(propertyName);
    }
    
    /**
     * Gets system property value with the given name.
     * @param propertyName system property name
     * @param defaultValue default value
     * @return system property value
     */
    public static String getSystemProperty(String propertyName, String defaultValue)
    {
        Assertions.checkForNullParameter(propertyName, "propertyName");
        return System.getProperty(propertyName,defaultValue);
    }
    
    /**
     * Gets system property value with the given name. If no property
     * found and default value is given, sets it.
     * @param propertyName system property name
     * @param defaultValue default value 
     * @return system property value
     */
    public static String getAndSetSystemProperty(String propertyName, String defaultValue)
    {
        Assertions.checkForNullParameter(propertyName, "propertyName");
        String value = getSystemProperty(propertyName);
        if(value == null)
        {
            System.setProperty(propertyName, defaultValue);
            value = defaultValue;
        }
        
        return value;
    }
    
    /**
     * Build chain of exception message.
     * @param e defne exception
     * @return full of the exception chain message
     */
    public static String getExceptionMessage(final DefneException e)
    {
        StringBuilder buffer = new StringBuilder("");
        if(e.getMessage() != null)
        {
            buffer.append(e.getMessage());
        }
        
        Throwable temp = e;
        Throwable cause = null;
        while((cause = temp.getCause()) != null)
        {
            buffer.append(":");
            buffer.append(cause.getMessage());
            
            temp = cause;
        }
        
        return buffer.toString();
    }
    
}
