/*
 * @Copyright 2010, MechSoft 
 * MechSoft, Mechanical and Software Solutions 
 * 
 * Licensed under the Apache License, Version 2.0 (the 
 * "License"); you may not use this file except in compliance 
 * with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.defne.utility.factories;

import java.util.HashMap;
import java.util.Map;

import org.defne.utility.Assertions;
import org.defne.utility.exception.DefneException;
import org.defne.utility.reflection.ReflectionUtils;

public final class SingletonProvider
{
    public static final String SERVICE_SCANNER_CLASS = "org.defne.service.annotation.ServiceScanner";
    
    public static final String DEFAULT_EMF_PROVIDER_CLASS = "org.defne.jpa.impl.DefaultEmfProvider";
    
    /**
     * Keys --> ClassLoaders
     * Values --> Maps of singleton class name with object
     */
    private static Map<ClassLoader, Map<String, Object>> singletonMap = new HashMap<ClassLoader, Map<String,Object>>();

    /**
     * No instantiate.
     */
    private SingletonProvider()
    {
        //No action
    }
    
    /**
     * Gets signelton instance.
     * @param singletonName singleton class name
     * @return singleton instance
     */
    public static Object getSingletonInstance(String singletonName)
    {
       return getSingletonInstance(singletonName, ReflectionUtils.getThreadClassLoader());
    }
    
    /**
     * Gets singleton instance for deployment.
     * @param singletonName singleton class name
     * @param classLoader classloader of the deployment
     * @return signelton instance for this deployment
     */
    public static Object getSingletonInstance(String singletonName, ClassLoader classLoader)
    {
        Object object = null;

        synchronized (singletonMap)
        {
            Map<String, Object> managerMap = singletonMap.get(classLoader);

            if (managerMap == null)
            {
                managerMap = new HashMap<String, Object>();
                singletonMap.put(classLoader, managerMap);
            }
            
            object = managerMap.get(singletonName);
            /* No singleton for this application, create one */
            if (object == null)
            {
                try
                {
                    //Load class
                    Class<?> clazz = ReflectionUtils.getClassFromName(singletonName);
                    
                    //Create instance
                    object = clazz.newInstance();
                    
                    //Save it
                    managerMap.put(singletonName, object);

                }
                catch (InstantiationException e)
                {
                    throw new DefneException("Unable to instantiate class : " + singletonName, e);
                }
                catch (IllegalAccessException e)
                {
                    throw new DefneException("Illegal access exception in creating instance with class : " + singletonName, e);
                }
            }
        }

        return object;
    }
    
    /**
     * Clear all deployment instances when the application is undeployed.
     * @param classloader of the deployment
     */
    public static void clearInstances(ClassLoader classLoader)
    {
        Assertions.checkForNull(classLoader, "classloader is null");
        synchronized (singletonMap)
        {
            singletonMap.remove(classLoader);
        }
    }    

}
