/*
 * @Copyright 2010, MechSoft 
 * MechSoft, Mechanical and Software Solutions 
 * 
 * Licensed under the Apache License, Version 2.0 (the 
 * "License"); you may not use this file except in compliance 
 * with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.defne.utility.log;

import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Implementation of the {@link ILogger} using
 * the platform JDK logging.
 * 
 * @version $Rev$ $Date$
 * 
 */
public class JDKLogger implements ILogger
{
    /**JDK Logging instance*/
    private final Logger logger;
    
    /**
     * Private Constructor.
     * @param clazz owner class of the logger
     */
    public JDKLogger(Class<?> clazz)
    {
        this.logger = Logger.getLogger(clazz.getName());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void debug(String message)
    {
       if(isDebugEnabled())
       {
           logger.fine(message);
       }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void debug(String message, Throwable error)
    {
        if(isDebugEnabled())
        {
            logger.log(Level.FINE, message, error);
        }
    }

    /**
     * {@inheritDoc}
     */    
    @Override
    public void error(String message)
    {
        if(isErrorEnabled())
        {
            logger.log(Level.SEVERE, message);
        }
        
    }

    /**
     * {@inheritDoc}
     */    
    @Override
    public void error(String message, Throwable error)
    {
        if(isErrorEnabled())
        {
            logger.log(Level.SEVERE, message ,error);
        }
    }

    /**
     * {@inheritDoc}
     */    
    @Override
    public void fatal(String message)
    {   
        error(message);
    }

    /**
     * {@inheritDoc}
     */    
    @Override
    public void fatal(String message, Throwable error)
    {
        error(message, error);
    }

    /**
     * {@inheritDoc}
     */    
    @Override
    public void info(String message)
    {
        if(isInfoEnabled())
        {
            logger.info(message);
        }
    }

    /**
     * {@inheritDoc}
     */    
    @Override
    public void info(String message, Throwable error)
    {
        if(isInfoEnabled())
        {
            logger.log(Level.INFO, message, error);
        }
    }

    /**
     * {@inheritDoc}
     */    
    @Override
    public boolean isDebugEnabled()
    {
        return this.logger.isLoggable(Level.FINE);
    }

    /**
     * {@inheritDoc}
     */    
    @Override
    public boolean isErrorEnabled()
    {
        return this.logger.isLoggable(Level.SEVERE);
    }

    /**
     * {@inheritDoc}
     */    
    @Override
    public boolean isFatalEnabled()
    {
        return isErrorEnabled();
    }

    /**
     * {@inheritDoc}
     */    
    @Override
    public boolean isInfoEnabled()
    {
        return this.logger.isLoggable(Level.INFO);
    }

    /**
     * {@inheritDoc}
     */    
    @Override
    public boolean isTraceEnabled()
    {
        return this.logger.isLoggable(Level.FINEST);
    }
    
    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isWarnEnabled()
    {
        return this.logger.isLoggable(Level.WARNING);
    }

    /**
     * {@inheritDoc}
     */    
    @Override
    public void trace(String message)
    {
        if(isTraceEnabled())
        {
            this.logger.log(Level.FINEST, message);
        }
    }

    /**
     * {@inheritDoc}
     */    
    @Override
    public void trace(String message, Throwable error)
    {
        if(isTraceEnabled())
        {
            this.logger.log(Level.FINEST, message, error);
        }
    }

    /**
     * {@inheritDoc}
     */    
    @Override
    public void warn(String message)
    {
        if(isWarnEnabled())
        {
            this.logger.log(Level.WARNING, message);
        }        
    }

    /**
     * {@inheritDoc}
     */    
    @Override
    public void warn(String message, Throwable error)
    {
        if(isWarnEnabled())
        {
            this.logger.log(Level.WARNING, message);
        }        
    }
    
    /**
     * {@inheritDoc}
     */
    @Override
    public void error(Throwable error)
    {
        this.logger.log(Level.SEVERE, "Exception without getting error message!", error);
    }

    /**
     * {@inheritDoc}
     */    
    @Override
    public void debug(Throwable error)
    {
        this.logger.log(Level.FINE, "Exception without getting error message!", error);
        
    }

    /**
     * {@inheritDoc}
     */    
    @Override
    public void fatal(Throwable error)
    {
        this.logger.log(Level.SEVERE, "Exception without getting error message!", error);
        
    }

    /**
     * {@inheritDoc}
     */    
    @Override
    public void info(Throwable error)
    {
        this.logger.log(Level.INFO, "Exception without getting error message!", error);
        
    }

    /**
     * {@inheritDoc}
     */    
    @Override
    public void trace(Throwable error)
    {
        this.logger.log(Level.FINEST, "Exception without getting error message!", error);
        
    }
    
    /**
     * {@inheritDoc}
     */
    @Override
    public void warn(Throwable error)
    {
        this.logger.log(Level.WARNING, "Exception without getting error message!", error);
    }
    
}
