/*
 * @Copyright 2010, MechSoft 
 * MechSoft, Mechanical and Software Solutions 
 * 
 * Licensed under the Apache License, Version 2.0 (the 
 * "License"); you may not use this file except in compliance 
 * with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.defne.utility.reflection;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.security.PrivilegedActionException;
import java.util.List;

import org.defne.utility.Assertions;
import org.defne.utility.Utility;
import org.defne.utility.exception.DefneException;
import org.defne.utility.log.ILogger;
import org.defne.utility.log.LoggerProvider;

/**
 * Reflection related utility class.
 * @version $Rev$ $Date$
 *
 */
public final class ReflectionUtils
{   
    /**Empty class array*/
    public static final Class<?>[] EMPTY = new Class<?>[0];

    /**Logger instance*/
    private static final ILogger logger = LoggerProvider.getLogProvider(ReflectionUtils.class);
        
    /**
     * Private constructor.
     */
    private ReflectionUtils()
    {
        throw new UnsupportedOperationException();
    }
    
    /**
     * Gets current thread classloader.
     * <p>
     * This class loader works that is explained in the
     * Java EE specifications.
     * </p>
     * @return current thread classloader
     */
    public static ClassLoader getThreadClassLoader()
    {
        ClassLoader loader;
        
        loader = AccessController.doPrivileged(new PrivilegedAction<ClassLoader>(){

            @Override
            public ClassLoader run()
            {
                try
                {
                    return Thread.currentThread().getContextClassLoader();
                    
                }catch(SecurityException e)
                {
                    return null;
                }
            }
            
        });
        
        return loader;
    }
    
    /**
     * Gets class instance.
     * @param <T> class type
     * @param clazz class of the object
     * @return the new instance of class
     */
    public static <T> Object getNewObject(Class<T> clazz)
    {
        try
        {
            return clazz.newInstance();
        }
        catch (InstantiationException e)
        {
            String error = "InstantiationException exception is thrown while " +
            		"creating instance of class : " + clazz;
            logger.error(error,e);
            throw new DefneException(error, e);
            
        }
        catch (IllegalAccessException e)
        {
            String error = "IllegalAccessException exception is thrown while " +
            		"creating instance of class : " + clazz;
            throw new DefneException(error, e);
        }
    }
    
    /**
     * Gets class constructor with the given parameters..
     * @param <T> class type
     * @param clazz class of the object
     * @param arguments contructor arguments
     * @return class constructor
     */
    @SuppressWarnings("unchecked")
    public static <T> Constructor<T> getConstructor(Class<T> clazz, List<Class<?>> arguments)
    {
        Assertions.checkForNullParameter(clazz, "clazz");
        Assertions.checkForNullParameter(arguments, "arguments");
        Constructor<?> constr = null;
        try
        {
            constr = clazz.getDeclaredConstructor(arguments.toArray(EMPTY));
        }
        catch (SecurityException e)
        {
            String error = "SecurityException exception is thrown by getting constructor";
            logger.error(error,e);
            throw new DefneException(error, e);
            
        }
        catch (NoSuchMethodException e)
        {
            String error = "NoSuchMethodException exception is thrown by getting constructor";            
            logger.error(error,e);
            throw new DefneException(error, e);
            
        }
        
        return (Constructor<T>)constr;
    }
    
    
    /**
     * Gets system class loader.
     * @return system class loader
     */
    public static ClassLoader getSystemClassLoader()
    {
        return ClassLoader.getSystemClassLoader();
    }
    
    /**
     * Calls the given object's method with supplied arguments.
     * @param instance object instance
     * @param method object method
     * @param arguments method arguments
     * @return the final result
     * @throws DefneException if any runtime exception occurs
     */
    public static Object invokeMethod(Object instance, Method method, Object[] arguments)
    {
        Object result = null;
        try
        {
            result = method.invoke(instance, arguments);
        }
        catch (IllegalArgumentException e)
        {
            String error = "IllegalArgument exception is thrown by calling method " + method + 
            " on instance " + instance + " with arguments " + arguments;
            logger.error(error,e);
            throw new DefneException(error, e);
        }
        catch (IllegalAccessException e)
        {
            String error = "IllegalAccessException exception is thrown by calling method " + method + 
            " on instance " + instance + " with arguments " + arguments;
            logger.error(error,e);
            throw new DefneException(error, e);
        }
        catch (InvocationTargetException e)
        {
            String error = "InvocationTargetException exception is thrown by calling method " + method + 
            " on instance " + instance + " with arguments " + arguments;
            Throwable e1 = Utility.getTargetException(e);
            logger.error(error,e1);
            throw new DefneException(error, e1);
        }
        
        return result;
    }
    
    
    /**
     * Load class with the given name and returns it.
     * If not succesfull, return null.
     * @param className class name
     * @return the class with the given name
     */
    public static Class<?> getClassFromName(String className)
    {
        Assertions.checkForNull(className, "className is null");
        ClassLoader loader = getThreadClassLoader();
        try
        {
            if(loader == null)
            {
                return Class.forName(className);
            }            
            else
            {
                return Class.forName(className, true, loader);
            }
        }catch(ClassNotFoundException e)
        {
            String error = "Class not found exception is thrown while " +
            "getting class : " + className; 
            logger.error(error, e);
            
            throw new DefneException(error, e);
        }
    }
    
    public static Object callConstructor(Constructor<?> constructor,Object[] arguments)
    { 
        try
        {
            return constructor.newInstance(arguments);
        }
        catch (IllegalArgumentException e)
        {
            String error = "IllegalArgumentException exception is thrown by calling constructor " + constructor + 
            " with arguments " + arguments;
            Throwable e1 = Utility.getTargetException(e);
            logger.error(error,e1);
            throw new DefneException(error, e1);
        }
        catch (InstantiationException e)
        {
            String error = "InstantiationException exception is thrown by calling constructor " + constructor + 
            " with arguments " + arguments;
            Throwable e1 = Utility.getTargetException(e);
            logger.error(error,e1);
            throw new DefneException(error, e1);
        }
        catch (IllegalAccessException e)
        {
            String error = "IllegalAccessException exception is thrown by calling constructor " + constructor + 
            " with arguments " + arguments;
            Throwable e1 = Utility.getTargetException(e);
            logger.error(error,e1);
            throw new DefneException(error, e1);
        }
        catch (InvocationTargetException e)
        {
            String error = "InvocationTargetException exception is thrown by calling constructor " + constructor + 
            " with arguments " + arguments;
            Throwable e1 = Utility.getTargetException(e);
            logger.error(error,e1);
            throw new DefneException(error, e1);
        }
    }
    
    public static void throwDefneExceptionFromPrivilege(PrivilegedActionException e)
    {
        DefneException exception = null;
        if(e.getCause() instanceof DefneException)
        {
            exception = (DefneException)e.getCause();
        }
        else
        {
            exception = new DefneException(e.getCause());
        }
        
        throw exception;
    }
    
    public static void main(String[] args)
    {
        System.out.println(ReflectionUtils.EMPTY);
    }
}
