/*
 * @Copyright 2010, MechSoft 
 * MechSoft, Mechanical and Software Solutions 
 * 
 * Licensed under the Apache License, Version 2.0 (the 
 * "License"); you may not use this file except in compliance 
 * with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.defne.utility.reflection;

import java.lang.reflect.Constructor;
import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.List;

import org.defne.utility.Utility;

public interface SecurityActions
{
    public class UTIL
    {
       public static SecurityActions getSecurityActions()
       {
          return System.getSecurityManager() == null ? NON_PRIVILEGED : PRIVILEGED;
       }
    }

    SecurityActions NON_PRIVILEGED = new SecurityActions()
    {

        @Override
        public Class<?> getClass(String className)
        {
            return ReflectionUtils.getClassFromName(className);
        }

        @Override
        public Constructor<?> getConstructor(Class<?> clazz, List<Class<?>> arguments)
        {
            return ReflectionUtils.getConstructor(clazz, arguments);
        }

        @Override
        public Object newInstance(Class<?> clazz)
        {
            return ReflectionUtils.getNewObject(clazz);
        }

        @Override
        public Object callConstructor(Constructor<?> constructor,Object[] arguments)
        {
            return ReflectionUtils.callConstructor(constructor, arguments);
        }

        @Override
        public String getSystemProperty(String systemProperty)
        {
            return Utility.getSystemProperty(systemProperty);
        }

        @Override
        public String getSystemProperty(String systemProperty, String defaultValue)
        {
            return Utility.getSystemProperty(systemProperty, defaultValue);
        }

        @Override
        public String getAndSetSystemProperty(String systemProperty, String value)
        {
            return Utility.getAndSetSystemProperty(systemProperty, value);
        }
        
    };
    
    SecurityActions PRIVILEGED = new SecurityActions()
    {

        @Override
        public Class<?> getClass(final String className)
        {
            PrivilegedExceptionAction<Class<?>> action = new PrivilegedExceptionAction<Class<?>>()
            {

                @Override
                public Class<?> run() throws Exception
                {
                    return ReflectionUtils.getClassFromName(className);
                }
                
            };
            
            try
            {
                return AccessController.doPrivileged(action);
            }
            catch (PrivilegedActionException e)
            {
                ReflectionUtils.throwDefneExceptionFromPrivilege(e);
            }
            
            return null;
        }

        @Override
        public Constructor<?> getConstructor(final Class<?> clazz, final List<Class<?>> arguments)
        {            
            PrivilegedExceptionAction<Constructor<?>> action = new PrivilegedExceptionAction<Constructor<?>>()
            {

                @Override
                public Constructor<?> run() throws Exception
                {
                    return ReflectionUtils.getConstructor(clazz, arguments);
                }
            };

            try
            {
                return AccessController.doPrivileged(action);
            }
            catch (PrivilegedActionException e)
            {
                ReflectionUtils.throwDefneExceptionFromPrivilege(e);
            }
            
            return null;
        }

        @Override
        public Object newInstance(final Class<?> clazz)
        {
            PrivilegedExceptionAction<Object> action = new PrivilegedExceptionAction<Object>()
            {

                @Override
                public Object run() throws Exception
                {
                    return ReflectionUtils.getNewObject(clazz);
                }
                
            };
           
            try
            {
                return AccessController.doPrivileged(action);
            }
            catch (PrivilegedActionException e)
            {
                ReflectionUtils.throwDefneExceptionFromPrivilege(e);
            }
            
            return null;
        }

        @Override
        public Object callConstructor(final Constructor<?> constructor, final Object[] arguments)
        {
            PrivilegedExceptionAction<Object> action = new PrivilegedExceptionAction<Object>()
            {

                @Override
                public Object run() throws Exception
                {                    
                    return ReflectionUtils.callConstructor(constructor, arguments);
                }
                
            };
            
            try
            {
                return AccessController.doPrivileged(action);
                
            }catch(PrivilegedActionException e)
            {
                ReflectionUtils.throwDefneExceptionFromPrivilege(e);
            }
            
            return null;
        }

        @Override
        public String getSystemProperty(final String systemProperty)
        {
            PrivilegedExceptionAction<String> action = new PrivilegedExceptionAction<String>()
            {

                @Override
                public String run() throws Exception
                {                    
                    return System.getProperty(systemProperty);
                }
                
            };
            
            try
            {
                return AccessController.doPrivileged(action);
                
            }catch(PrivilegedActionException e)
            {
                ReflectionUtils.throwDefneExceptionFromPrivilege(e);
            }
            
            return null;
        }

        @Override
        public String getSystemProperty(final String systemProperty, final String defaultValue)
        {
            PrivilegedExceptionAction<String> action = new PrivilegedExceptionAction<String>()
            {

                @Override
                public String run() throws Exception
                {                    
                    return System.getProperty(systemProperty, defaultValue);
                }
                
            };
            
            try
            {
                return AccessController.doPrivileged(action);
                
            }catch(PrivilegedActionException e)
            {
                ReflectionUtils.throwDefneExceptionFromPrivilege(e);
            }
            
            return null;
        }

        @Override
        public String getAndSetSystemProperty(final String systemProperty, final String value)
        {
            PrivilegedExceptionAction<String> action = new PrivilegedExceptionAction<String>()
            {

                @Override
                public String run() throws Exception
                {                    
                    return Utility.getAndSetSystemProperty(systemProperty, value);
                }
                
            };
            
            try
            {
               return AccessController.doPrivileged(action);
                
            }catch(PrivilegedActionException e)
            {
                ReflectionUtils.throwDefneExceptionFromPrivilege(e);
            }
            
            return null;
        }
        
    };    
    

    public Constructor<?> getConstructor(Class<?> clazz, List<Class<?>> arguments);
    
    public Object newInstance(Class<?> clazz);
    
    public Class<?> getClass(String className);
    
    public Object callConstructor(Constructor<?> constructor,Object[] arguments);
    
    public String getSystemProperty(String systemProperty);
    
    public String getSystemProperty(String systemProperty, String defaultValue);
    
    public String getAndSetSystemProperty(String systemProperty, String value);
}
