/*
 * Decompiled with CFR 0.152.
 */
package org.deltafi.common.trace;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.time.OffsetDateTime;
import java.util.ArrayList;
import java.util.Map;
import java.util.Objects;
import java.util.Queue;
import java.util.UUID;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import javax.annotation.PreDestroy;
import org.deltafi.common.properties.ZipkinProperties;
import org.deltafi.common.trace.DeltafiSpan;
import org.deltafi.common.trace.ZipkinRestClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZipkinService {
    private static final Logger log = LoggerFactory.getLogger(ZipkinService.class);
    private static final String ROOT_PARENT_ID = "0";
    private static final String ROOT_SPAN_NAME = "deltafile-flow";
    private static final String SERVER = "SERVER";
    private static final String DID = "did";
    private static final String FLOW = "flow";
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    private static final Queue<DeltafiSpan> SPANS_QUEUE = new ConcurrentLinkedQueue<DeltafiSpan>();
    private final ZipkinRestClient zipkinRestClient;
    private final ZipkinProperties zipkinProperties;
    private final ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor();

    public ZipkinService(ZipkinProperties zipkinProperties) {
        this.zipkinProperties = zipkinProperties;
        this.zipkinRestClient = new ZipkinRestClient(zipkinProperties.getUrl());
        if (zipkinProperties.isEnabled()) {
            this.executor.scheduleAtFixedRate(this::sendQueuedSpans, zipkinProperties.getSendInitialDelayMs(), zipkinProperties.getSendPeriodMs(), TimeUnit.MILLISECONDS);
        }
    }

    @PreDestroy
    public void shutdown() {
        this.executor.shutdown();
    }

    public void createAndSendRootSpan(String did, OffsetDateTime created, String fileName, String flow) {
        DeltafiSpan rootSpan = DeltafiSpan.newSpanBuilder().traceId(this.traceId(did)).localEndpoint(fileName).timestamp(created.toInstant()).id(this.hexIdFromUuid(did)).name(ROOT_SPAN_NAME).parentId(ROOT_PARENT_ID).tags(Map.of(DID, did, FLOW, flow)).kind(SERVER).build();
        this.markSpanComplete(rootSpan);
    }

    public DeltafiSpan createChildSpan(String did, String actionName, String fileName, String flow) {
        return this.createChildSpan(did, actionName, fileName, flow, null);
    }

    public DeltafiSpan createChildSpan(String did, String actionName, String fileName, String flow, OffsetDateTime created) {
        DeltafiSpan.Builder builder = DeltafiSpan.newSpanBuilder().localEndpoint(fileName).traceId(this.traceId(did)).id(this.longToHex(this.uniqueId())).name(actionName).parentId(this.hexIdFromUuid(did)).tags(Map.of(DID, did, FLOW, flow)).kind(SERVER);
        if (created != null) {
            builder.timestamp(created.toInstant());
        }
        return builder.build();
    }

    public void markSpanComplete(DeltafiSpan span) {
        if (this.zipkinProperties.isEnabled()) {
            span.endSpan();
            SPANS_QUEUE.add(span);
            if (SPANS_QUEUE.size() >= this.zipkinProperties.getMaxBatchSize()) {
                this.sendQueuedSpans();
            }
        }
    }

    public void sendQueuedSpans() {
        if (SPANS_QUEUE.isEmpty()) {
            return;
        }
        String spansAsJson = this.spansToJson();
        if (Objects.nonNull(spansAsJson) && !spansAsJson.isBlank()) {
            this.zipkinRestClient.sendSpan(spansAsJson);
        }
    }

    String spansToJson() {
        ArrayList<DeltafiSpan> toSend = new ArrayList<DeltafiSpan>();
        for (int cnt = 0; cnt < this.zipkinProperties.getMaxBatchSize() && !SPANS_QUEUE.isEmpty(); ++cnt) {
            toSend.add(SPANS_QUEUE.poll());
        }
        try {
            return OBJECT_MAPPER.writeValueAsString(toSend);
        }
        catch (JsonProcessingException e) {
            log.error("Failed to send the following spans to Zipkin, {}", toSend, (Object)e);
            return null;
        }
    }

    private String traceId(String did) {
        return did.replace("-", "");
    }

    private String hexIdFromUuid(String uuidString) {
        UUID uuid = UUID.fromString(uuidString);
        long mostSignificantBits = uuid.getMostSignificantBits();
        return this.longToHex(mostSignificantBits);
    }

    private String longToHex(long longValue) {
        return Long.toHexString(longValue);
    }

    private long uniqueId() {
        return ThreadLocalRandom.current().nextLong();
    }

    public boolean isEnabled() {
        return this.zipkinProperties.isEnabled();
    }
}

