/*
 * Decompiled with CFR 0.152.
 */
package org.dflib.parquet.write;

import java.nio.ByteBuffer;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.function.Consumer;
import org.apache.parquet.io.api.Binary;
import org.apache.parquet.io.api.RecordConsumer;
import org.dflib.parquet.write.BigDecimalWrite;
import org.dflib.parquet.write.ColumnMeta;
import org.dflib.parquet.write.DataFrameSchema;
import org.dflib.parquet.write.InstantWrite;
import org.dflib.parquet.write.LocalDateTimeWrite;
import org.dflib.parquet.write.WriteConfiguration;
import org.dflib.row.RowProxy;

class RowWriter {
    private final RecordConsumer recordConsumer;
    private final WriteConfiguration writeConfiguration;
    private final List<Consumer<RowProxy>> fieldWriters = new ArrayList<Consumer<RowProxy>>();

    public RowWriter(RecordConsumer recordConsumer, WriteConfiguration writeConfiguration, DataFrameSchema schema) {
        this.recordConsumer = recordConsumer;
        this.writeConfiguration = writeConfiguration;
        for (ColumnMeta column : schema.getColumns()) {
            Consumer<RowProxy> writer = this.buildWriter(column);
            this.fieldWriters.add(writer);
        }
    }

    public void write(RowProxy record) {
        this.recordConsumer.startMessage();
        for (Consumer<RowProxy> fieldWriter : this.fieldWriters) {
            fieldWriter.accept(record);
        }
        this.recordConsumer.endMessage();
    }

    private Consumer<RowProxy> buildWriter(ColumnMeta column) {
        String name = column.getInferredTypeName();
        int idx = column.getIndex();
        switch (name) {
            case "int": {
                return new PrimitiveFieldWriter(column, row -> this.recordConsumer.addInteger(row.getInt(idx)));
            }
            case "java.lang.Integer": {
                return new FieldWriter(column, v -> this.recordConsumer.addInteger(((Integer)v).intValue()));
            }
            case "long": {
                return new PrimitiveFieldWriter(column, row -> this.recordConsumer.addLong(row.getLong(idx)));
            }
            case "java.lang.Long": {
                return new FieldWriter(column, v -> this.recordConsumer.addLong(((Long)v).longValue()));
            }
            case "java.lang.Byte": 
            case "java.lang.Short": {
                return new FieldWriter(column, v -> this.recordConsumer.addInteger(((Number)v).intValue()));
            }
            case "java.lang.Float": {
                return new FieldWriter(column, v -> this.recordConsumer.addFloat(((Float)v).floatValue()));
            }
            case "double": {
                return new PrimitiveFieldWriter(column, row -> this.recordConsumer.addDouble(row.getDouble(idx)));
            }
            case "java.lang.Double": {
                return new FieldWriter(column, v -> this.recordConsumer.addDouble(((Double)v).doubleValue()));
            }
            case "boolean": {
                return new PrimitiveFieldWriter(column, row -> this.recordConsumer.addBoolean(row.getBool(idx)));
            }
            case "java.lang.Boolean": {
                return new FieldWriter(column, v -> this.recordConsumer.addBoolean(((Boolean)v).booleanValue()));
            }
            case "java.lang.String": {
                return new FieldWriter(column, v -> this.recordConsumer.addBinary(Binary.fromString((String)((String)v))));
            }
            case "java.util.UUID": {
                return new FieldWriter(column, v -> this.recordConsumer.addBinary(RowWriter.uuidToBinary(v)));
            }
            case "java.math.BigDecimal": {
                BigDecimalWrite bigDecimalWrite = new BigDecimalWrite(this.writeConfiguration.getDecimalConfig());
                return new FieldWriter(column, v -> bigDecimalWrite.write(this.recordConsumer, v));
            }
            case "java.time.LocalDate": {
                return new FieldWriter(column, v -> this.recordConsumer.addInteger((int)((LocalDate)v).toEpochDay()));
            }
            case "java.time.LocalTime": {
                switch (this.writeConfiguration.getTimeUnit()) {
                    case MILLIS: {
                        return new FieldWriter(column, v -> this.recordConsumer.addInteger((int)(((LocalTime)v).toNanoOfDay() / 1000000L)));
                    }
                    case MICROS: {
                        return new FieldWriter(column, v -> this.recordConsumer.addLong(((LocalTime)v).toNanoOfDay() / 1000L));
                    }
                    case NANOS: {
                        return new FieldWriter(column, v -> this.recordConsumer.addLong(((LocalTime)v).toNanoOfDay()));
                    }
                }
                throw new IllegalArgumentException("Invalid " + this.writeConfiguration.getTimeUnit());
            }
            case "java.time.LocalDateTime": {
                LocalDateTimeWrite.LocalDateTimeToLong localDateMapper = LocalDateTimeWrite.getLocalDateTimeMapper(this.writeConfiguration.getTimeUnit());
                return new FieldWriter(column, v -> this.recordConsumer.addLong(localDateMapper.map((LocalDateTime)v)));
            }
            case "java.time.Instant": {
                InstantWrite.InstantToLong instantMapper = InstantWrite.getInstantMapper(this.writeConfiguration.getTimeUnit());
                return new FieldWriter(column, v -> this.recordConsumer.addLong(instantMapper.map((Instant)v)));
            }
        }
        if (column.isEnum()) {
            return new FieldWriter(column, v -> this.recordConsumer.addBinary(Binary.fromString((String)((Enum)v).name())));
        }
        throw new IllegalArgumentException("Unsupported type " + name);
    }

    private static Binary uuidToBinary(Object value) {
        UUID uuid = (UUID)value;
        byte[] arr = new byte[16];
        ByteBuffer bb = ByteBuffer.wrap(arr);
        bb.putLong(uuid.getMostSignificantBits());
        bb.putLong(uuid.getLeastSignificantBits());
        return Binary.fromConstantByteArray((byte[])arr);
    }

    private class FieldWriter
    implements Consumer<RowProxy> {
        private final String fieldName;
        private final int idx;
        private final Consumer<Object> valueConsumer;

        FieldWriter(ColumnMeta column, Consumer<Object> valueConsumer) {
            this.fieldName = column.getColumnName();
            this.idx = column.getIndex();
            this.valueConsumer = valueConsumer;
        }

        @Override
        public void accept(RowProxy rowProxy) {
            Object value = rowProxy.get(this.idx);
            if (value != null) {
                RowWriter.this.recordConsumer.startField(this.fieldName, this.idx);
                this.valueConsumer.accept(value);
                RowWriter.this.recordConsumer.endField(this.fieldName, this.idx);
            }
        }
    }

    private class PrimitiveFieldWriter
    implements Consumer<RowProxy> {
        private final String fieldName;
        private final int idx;
        private final Consumer<RowProxy> rowConsumer;

        PrimitiveFieldWriter(ColumnMeta column, Consumer<RowProxy> rowConsumer) {
            this.fieldName = column.getColumnName();
            this.idx = column.getIndex();
            this.rowConsumer = rowConsumer;
        }

        @Override
        public void accept(RowProxy rowProxy) {
            RowWriter.this.recordConsumer.startField(this.fieldName, this.idx);
            this.rowConsumer.accept(rowProxy);
            RowWriter.this.recordConsumer.endField(this.fieldName, this.idx);
        }
    }
}

