/*
 * Decompiled with CFR 0.152.
 */
package org.dihedron.crypto.certificates;

import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.GeneralSecurityException;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PublicKey;
import java.security.SignatureException;
import java.security.cert.CertPathBuilder;
import java.security.cert.CertPathBuilderException;
import java.security.cert.CertSelector;
import java.security.cert.CertStore;
import java.security.cert.CertStoreParameters;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CollectionCertStoreParameters;
import java.security.cert.PKIXBuilderParameters;
import java.security.cert.PKIXCertPathBuilderResult;
import java.security.cert.TrustAnchor;
import java.security.cert.X509CertSelector;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1Integer;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.ess.ESSCertID;
import org.bouncycastle.asn1.ess.ESSCertIDv2;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.IssuerSerial;
import org.bouncycastle.cert.jcajce.JcaX509CertificateHolder;
import org.dihedron.core.License;
import org.dihedron.crypto.CryptoService;
import org.dihedron.crypto.constants.DigestAlgorithm;
import org.dihedron.crypto.crl.CRL;
import org.dihedron.crypto.exceptions.CertificateVerificationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@License(copyright="Copyright (c) 2012-2014 Andrea Funto', Svetlin Nakov")
public final class Certificates
extends CryptoService {
    private static final Logger logger = LoggerFactory.getLogger(Certificates.class);

    public static boolean isSignatureX509Certificate(X509Certificate certificate) {
        boolean[] usage = certificate.getKeyUsage();
        for (KeyUsage u : KeyUsage.values()) {
            if (!usage[u.ordinal()]) continue;
            logger.trace("bit '{}' set", (Object)u.name());
        }
        if (usage != null && usage[KeyUsage.keyEncipherment.ordinal()] && usage[KeyUsage.digitalSignature.ordinal()]) {
            logger.trace("this is a signing certificate (bits set)");
            return true;
        }
        return false;
    }

    public static boolean isNonRepudiationX509Certificate(X509Certificate certificate) {
        boolean[] usage = certificate.getKeyUsage();
        for (KeyUsage u : KeyUsage.values()) {
            if (!usage[u.ordinal()]) continue;
            logger.trace("bit '{}' set", (Object)u.name());
        }
        if (usage != null && usage[KeyUsage.nonRepudiation.ordinal()]) {
            logger.trace("this is a non repudiation certificate (bits set)");
            return true;
        }
        return false;
    }

    public static boolean isSelfSigned(X509Certificate certificate) throws CertificateException, NoSuchAlgorithmException, NoSuchProviderException {
        try {
            PublicKey key = certificate.getPublicKey();
            certificate.verify(key);
            return true;
        }
        catch (InvalidKeyException | SignatureException e) {
            return false;
        }
    }

    public static boolean hasCriticalExtension(X509Certificate certificate, String oid) {
        logger.debug("looking for critical extension OID '{}'...", (Object)oid);
        Set<String> extensions = certificate.getCriticalExtensionOIDs();
        for (String extension : extensions) {
            logger.trace("... analysing critical extension '{}'", (Object)extension);
            if (!extension.contains(oid)) continue;
            logger.trace("... OID found!");
            return true;
        }
        return false;
    }

    public static PKIXCertPathBuilderResult verifyCertificate(X509Certificate certificate, Collection<X509Certificate> additionalCerts) throws CertificateVerificationException {
        try {
            logger.trace("verifying certificate:\n{}", (Object)certificate);
            if (Certificates.isSelfSigned(certificate)) {
                logger.error("certificate is self signed");
                throw new CertificateVerificationException("the certificate is self-signed");
            }
            HashSet<X509Certificate> trustedRootCerts = new HashSet<X509Certificate>();
            HashSet<X509Certificate> intermediateCerts = new HashSet<X509Certificate>();
            for (X509Certificate additionalCert : additionalCerts) {
                if (Certificates.isSelfSigned(additionalCert)) {
                    trustedRootCerts.add(additionalCert);
                    continue;
                }
                intermediateCerts.add(additionalCert);
            }
            PKIXCertPathBuilderResult verifiedCertChain = Certificates.verifyCertificate(certificate, trustedRootCerts, intermediateCerts);
            logger.info("certification chain verified");
            CRL.verifyCertificateCRLs(certificate);
            logger.info("CRL verified");
            return verifiedCertChain;
        }
        catch (CertPathBuilderException e) {
            logger.error("error building certification path for " + certificate.getSubjectX500Principal(), (Throwable)e);
            throw new CertificateVerificationException("Error building certification path: " + certificate.getSubjectX500Principal(), e);
        }
        catch (NoSuchAlgorithmException | NoSuchProviderException | CertificateException e) {
            logger.error("error verifying certificate " + certificate.getSubjectX500Principal(), (Throwable)e);
            throw new CertificateVerificationException("Error verifying the certificate: " + certificate.getSubjectX500Principal(), e);
        }
        catch (GeneralSecurityException e) {
            logger.error("error verifying certificate " + certificate.getSubjectX500Principal(), (Throwable)e);
            throw new CertificateVerificationException("Error verifying the certificate: " + certificate.getSubjectX500Principal(), e);
        }
    }

    private static PKIXCertPathBuilderResult verifyCertificate(X509Certificate certificate, Collection<X509Certificate> trustedRootCerts, Collection<X509Certificate> intermediateCerts) throws InvalidAlgorithmParameterException, NoSuchAlgorithmException, NoSuchProviderException, CertPathBuilderException {
        X509CertSelector selector = new X509CertSelector();
        selector.setCertificate(certificate);
        HashSet<TrustAnchor> trustAnchors = new HashSet<TrustAnchor>();
        for (X509Certificate trustedRootCert : trustedRootCerts) {
            trustAnchors.add(new TrustAnchor(trustedRootCert, null));
        }
        PKIXBuilderParameters pkixParams = new PKIXBuilderParameters(trustAnchors, (CertSelector)selector);
        pkixParams.setRevocationEnabled(false);
        CertStore intermediateCertStore = CertStore.getInstance("Collection", (CertStoreParameters)new CollectionCertStoreParameters(intermediateCerts), "BC");
        pkixParams.addCertStore(intermediateCertStore);
        CertPathBuilder builder = CertPathBuilder.getInstance("PKIX", "BC");
        return (PKIXCertPathBuilderResult)builder.build(pkixParams);
    }

    public static ASN1Primitive getExtensionValue(X509Certificate certificate, String oid) throws IOException {
        byte[] bytes = certificate.getExtensionValue(oid);
        if (bytes == null) {
            return null;
        }
        ASN1InputStream stream = new ASN1InputStream((InputStream)new ByteArrayInputStream(bytes));
        ASN1OctetString octets = (ASN1OctetString)stream.readObject();
        stream = new ASN1InputStream((InputStream)new ByteArrayInputStream(octets.getOctets()));
        return stream.readObject();
    }

    public static IssuerSerial makeIssuerSerial(X509Certificate x509certificate) throws CertificateEncodingException, IOException {
        JcaX509CertificateHolder holder = new JcaX509CertificateHolder(x509certificate);
        ASN1Integer serial = new ASN1Integer(holder.getSerialNumber());
        logger.debug("signer's certificate serial no.: '{}'", (Object)serial);
        X500Name issuer = holder.getIssuer();
        logger.debug("signer's certificate principal: '{}'", (Object)issuer);
        return new IssuerSerial(issuer, holder.getSerialNumber());
    }

    public static ESSCertID makeESSCertIdV1(X509Certificate x509certificate, IssuerSerial issuerSerial, DigestAlgorithm digestAlgorithm) throws NoSuchAlgorithmException, CertificateEncodingException {
        logger.info("adding signing certificate v1 to signed attributes");
        MessageDigest digest = MessageDigest.getInstance(digestAlgorithm.getAsn1Id());
        ESSCertID essCertIdV1 = new ESSCertID(digest.digest(x509certificate.getEncoded()), issuerSerial);
        return essCertIdV1;
    }

    public static ESSCertIDv2[] makeESSCertIdV2(X509Certificate x509certificate, IssuerSerial issuerSerial, DigestAlgorithm digestAlgorithm) throws NoSuchAlgorithmException, CertificateEncodingException {
        logger.info("adding signing certificate v2 to signed attributes");
        MessageDigest digest = MessageDigest.getInstance(digestAlgorithm.getAsn1Id());
        ESSCertIDv2 essCertIdv2 = new ESSCertIDv2(digest.digest(x509certificate.getEncoded()), issuerSerial);
        ESSCertIDv2[] essCertIdv2s = new ESSCertIDv2[]{essCertIdv2};
        return essCertIdv2s;
    }

    public static CertStore makeCertificateStore(Certificate certificate) throws InvalidAlgorithmParameterException, NoSuchAlgorithmException, NoSuchProviderException {
        ArrayList<Certificate> certificates = new ArrayList<Certificate>();
        certificates.add(certificate);
        CertStore store = CertStore.getInstance("Collection", (CertStoreParameters)new CollectionCertStoreParameters(certificates), "BC");
        return store;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean writeToFile(Certificate certificate, String filename) {
        boolean result = false;
        OutputStream fos = null;
        try {
            byte[] data = certificate.getEncoded();
            fos = new FileOutputStream(filename);
            ((FileOutputStream)fos).write(data);
            result = true;
        }
        catch (CertificateEncodingException e) {
            logger.error("certificate encoding error", (Throwable)e);
        }
        catch (FileNotFoundException e) {
            logger.error("file not found", (Throwable)e);
        }
        catch (IOException e) {
            logger.error("error writing certificate to disk", (Throwable)e);
        }
        finally {
            try {
                fos.flush();
                ((FileOutputStream)fos).close();
            }
            catch (IOException e) {
                logger.error("error closing output stream", (Throwable)e);
            }
        }
        return result;
    }

    private Certificates() {
    }

    private static enum KeyUsage {
        digitalSignature,
        nonRepudiation,
        keyEncipherment,
        dataEncipherment,
        keyAgreement,
        keyCertSign,
        cRLSign,
        encipherOnly,
        decipherOnly;

    }
}

