/*
 * Decompiled with CFR 0.152.
 */
package htsjdk.samtools;

import htsjdk.samtools.BAMFileSpan;
import htsjdk.samtools.BAMIndexContent;
import htsjdk.samtools.BAMIndexMetaData;
import htsjdk.samtools.BAMIndexWriter;
import htsjdk.samtools.BAMIndexer;
import htsjdk.samtools.BinaryBAMIndexWriter;
import htsjdk.samtools.BinningIndexBuilder2;
import htsjdk.samtools.BinningIndexContent;
import htsjdk.samtools.CachingBAMFileIndex;
import htsjdk.samtools.Chunk;
import htsjdk.samtools.SAMException;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMFileSource;
import htsjdk.samtools.SAMRecord;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.SamReader;
import htsjdk.samtools.TextualBAMIndexWriter;
import htsjdk.samtools.util.Log;
import java.io.File;
import java.io.OutputStream;
import java.nio.file.Path;
import java.util.function.Function;

public class BAMIndexer2 {
    private final int numReferences;
    private final BAMIndexWriter outputWriter;
    private int currentReference = 0;
    private final BAMIndexBuilder indexBuilder;
    private static final Log log = Log.getInstance(BAMIndexer.class);

    public BAMIndexer2(Path output, SAMFileHeader fileHeader) {
        this(fileHeader, (Integer numRefs) -> new BinaryBAMIndexWriter(numRefs.intValue(), output));
    }

    public BAMIndexer2(File output, SAMFileHeader fileHeader) {
        this(output.toPath(), fileHeader);
    }

    public BAMIndexer2(OutputStream output, SAMFileHeader fileHeader) {
        this(fileHeader, (Integer numRefs) -> new BinaryBAMIndexWriter(numRefs.intValue(), output));
    }

    private BAMIndexer2(SAMFileHeader fileHeader, Function<Integer, BinaryBAMIndexWriter> createWriter) {
        if (fileHeader.getSortOrder() != SAMFileHeader.SortOrder.coordinate) {
            if (fileHeader.getSortOrder() == SAMFileHeader.SortOrder.unsorted) {
                log.warn(new Object[]{"For indexing, the BAM file is required to be coordinate sorted. Attempting to index \"unsorted\" BAM file."});
            } else {
                throw new SAMException("Indexing requires a coordinate-sorted input BAM.");
            }
        }
        this.numReferences = fileHeader.getSequenceDictionary().size();
        this.indexBuilder = new BAMIndexBuilder(fileHeader.getSequenceDictionary());
        this.outputWriter = (BAMIndexWriter)createWriter.apply(this.numReferences);
    }

    public void processAlignment(SAMRecord rec) {
        try {
            int reference = rec.getReferenceIndex();
            if (reference != -1 && reference != this.currentReference) {
                this.advanceToReference(reference);
            }
            this.indexBuilder.processAlignment(rec);
        }
        catch (Exception e) {
            throw new SAMException("Exception creating BAM index for record " + rec, (Throwable)e);
        }
    }

    public void finish() {
        this.advanceToReference(this.numReferences);
        this.outputWriter.writeNoCoordinateRecordCount(Long.valueOf(this.indexBuilder.getNoCoordinateRecordCount()));
        this.outputWriter.close();
    }

    private void advanceToReference(int nextReference) {
        while (this.currentReference < nextReference) {
            BAMIndexContent content = this.indexBuilder.processReference(this.currentReference);
            this.outputWriter.writeReference(content);
            ++this.currentReference;
            if (this.currentReference >= this.numReferences) continue;
            this.indexBuilder.startNewReference();
        }
    }

    public static void createAndWriteIndex(File input, File output, boolean textOutput) {
        CachingBAMFileIndex existingIndex = new CachingBAMFileIndex(input, null);
        int n_ref = existingIndex.getNumberOfReferences();
        Object outputWriter = textOutput ? new TextualBAMIndexWriter(n_ref, output) : new BinaryBAMIndexWriter(n_ref, output);
        try {
            for (int i = 0; i < n_ref; ++i) {
                outputWriter.writeReference(existingIndex.getQueryResults(i));
            }
            outputWriter.writeNoCoordinateRecordCount(existingIndex.getNoCoordinateCount());
            outputWriter.close();
        }
        catch (Exception e) {
            throw new SAMException("Exception creating BAM index", (Throwable)e);
        }
    }

    public static void createIndex(SamReader reader, Path output) {
        BAMIndexer2.createIndex(reader, output, null);
    }

    public static void createIndex(SamReader reader, File output) {
        BAMIndexer2.createIndex(reader, output.toPath(), null);
    }

    public static void createIndex(SamReader reader, Path output, Log log) {
        BAMIndexer indexer = new BAMIndexer(output, reader.getFileHeader());
        long totalRecords = 0L;
        for (SAMRecord rec : reader) {
            if (++totalRecords % 1000000L == 0L && null != log) {
                log.info(new Object[]{totalRecords + " reads processed ..."});
            }
            indexer.processAlignment(rec);
        }
        indexer.finish();
    }

    public static void createIndex(SamReader reader, File output, Log log) {
        BAMIndexer2.createIndex(reader, output.toPath(), log);
    }

    private class BAMIndexBuilder {
        private final SAMSequenceDictionary sequenceDictionary;
        private BinningIndexBuilder2 binningIndexBuilder;
        private int currentReference = -1;
        private final BAMIndexMetaData indexStats = new BAMIndexMetaData();

        BAMIndexBuilder(SAMSequenceDictionary sequenceDictionary) {
            this.sequenceDictionary = sequenceDictionary;
            if (!sequenceDictionary.isEmpty()) {
                this.startNewReference();
            }
        }

        public void processAlignment(final SAMRecord rec) {
            this.indexStats.recordMetaData(rec);
            if (rec.getAlignmentStart() == 0) {
                return;
            }
            int reference = rec.getReferenceIndex();
            if (reference != this.currentReference) {
                throw new SAMException("Unexpected reference " + reference + " when constructing index for " + this.currentReference + " for record " + rec);
            }
            this.binningIndexBuilder.processFeature(new BinningIndexBuilder2.FeatureToBeIndexed(){

                @Override
                public int getStart() {
                    return rec.getAlignmentStart();
                }

                @Override
                public int getEnd() {
                    return rec.getAlignmentEnd();
                }

                @Override
                public Integer getIndexingBin() {
                    Integer binNumber = rec.getIndexingBin();
                    return binNumber == null ? rec.computeIndexingBin() : binNumber.intValue();
                }

                @Override
                public Chunk getChunk() {
                    SAMFileSource source = rec.getFileSource();
                    if (source == null) {
                        throw new SAMException("No source (virtual file offsets); needed for indexing on BAM Record " + rec);
                    }
                    return ((BAMFileSpan)source.getFilePointer()).getSingleChunk();
                }
            });
        }

        public BAMIndexContent processReference(int reference) {
            if (reference != this.currentReference) {
                throw new SAMException("Unexpected reference " + reference + " when constructing index for " + this.currentReference);
            }
            BinningIndexContent indexContent = this.binningIndexBuilder.generateIndexContent();
            if (indexContent == null) {
                return null;
            }
            return new BAMIndexContent(indexContent.getReferenceSequence(), indexContent.getBins(), this.indexStats, indexContent.getLinearIndex());
        }

        public long getNoCoordinateRecordCount() {
            return this.indexStats.getNoCoordinateRecordCount();
        }

        void startNewReference() {
            ++this.currentReference;
            this.indexStats.newReference();
            this.binningIndexBuilder = new BinningIndexBuilder2(this.currentReference, this.sequenceDictionary.getSequence(this.currentReference).getSequenceLength());
        }
    }
}

