package org.dreamcat.common.x.asm;

import java.util.Collection;
import java.util.Map;
import javassist.ByteArrayClassPath;
import javassist.CannotCompileException;
import javassist.ClassPool;
import javassist.CtClass;
import javassist.NotFoundException;
import javassist.bytecode.AnnotationsAttribute;
import javassist.bytecode.ClassFile;
import javassist.bytecode.ConstPool;
import javassist.bytecode.annotation.Annotation;
import javassist.bytecode.annotation.ArrayMemberValue;
import javassist.bytecode.annotation.BooleanMemberValue;
import javassist.bytecode.annotation.ByteMemberValue;
import javassist.bytecode.annotation.CharMemberValue;
import javassist.bytecode.annotation.ClassMemberValue;
import javassist.bytecode.annotation.DoubleMemberValue;
import javassist.bytecode.annotation.EnumMemberValue;
import javassist.bytecode.annotation.FloatMemberValue;
import javassist.bytecode.annotation.IntegerMemberValue;
import javassist.bytecode.annotation.LongMemberValue;
import javassist.bytecode.annotation.MemberValue;
import javassist.bytecode.annotation.ShortMemberValue;
import javassist.bytecode.annotation.StringMemberValue;
import org.dreamcat.common.util.ReflectUtil;

/**
 * Create by tuke on 2020/5/28
 */
public final class CtClassUtil {

    private CtClassUtil() {
    }

    public static CtClass createCtClass(String className, byte[] classBytes)
            throws NotFoundException {
        classPool.insertClassPath(new ByteArrayClassPath(className, classBytes));
        return classPool.get(className);
    }

    // ==== ==== ==== ====    ==== ==== ==== ====    ==== ==== ==== ====

    public static AnnotationsAttribute createAnnotationsAttribute(
            CtClass cc, Map<String, Map<String, Object>> annotations) {
        return createAnnotationsAttribute(cc.getClassFile(), annotations);
    }

    public static AnnotationsAttribute createAnnotationsAttribute(
            ClassFile classFile, Map<String, Map<String, Object>> annotations) {
        return createAnnotationsAttribute(classFile.getConstPool(), annotations);
    }

    public static AnnotationsAttribute createAnnotationsAttribute(
            ConstPool constPool, Map<String, Map<String, Object>> annotations) {
        AnnotationsAttribute attribute = new AnnotationsAttribute(
                constPool, AnnotationsAttribute.visibleTag);
        for (Map.Entry<String, Map<String, Object>> annotationEntry : annotations.entrySet()) {
            String annotationClassName = annotationEntry.getKey();
            Map<String, Object> annotationMembers = annotationEntry.getValue();
            Annotation annotation = new Annotation(annotationClassName, constPool);
            for (Map.Entry<String, Object> entry : annotationMembers.entrySet()) {
                String name = entry.getKey();
                Object value = entry.getValue();
                annotation.addMemberValue(name, castToMemberValue(value, constPool));
            }
            attribute.addAnnotation(annotation);
        }
        return attribute;
    }

    public static MemberValue castToMemberValue(Object value, ConstPool constPool) {
        if (value instanceof String) {
            return new StringMemberValue((String) value, constPool);
        } else if (value instanceof Byte) {
            return new ByteMemberValue((Byte) value, constPool);
        } else if (value instanceof Short) {
            return new ShortMemberValue((Short) value, constPool);
        } else if (value instanceof Character) {
            return new CharMemberValue((Character) value, constPool);
        } else if (value instanceof Integer) {
            return new IntegerMemberValue(constPool, (Integer) value);
        } else if (value instanceof Long) {
            return new LongMemberValue((Long) value, constPool);
        } else if (value instanceof Float) {
            return new FloatMemberValue((Float) value, constPool);
        } else if (value instanceof Double) {
            return new DoubleMemberValue((Double) value, constPool);
        } else if (value instanceof Boolean) {
            return new BooleanMemberValue((Boolean) value, constPool);
        } else if (value instanceof Class) {
            return new ClassMemberValue(((Class<?>) value).getCanonicalName(), constPool);
        } else if (value instanceof Enum) {
            Enum<?> en = (Enum<?>) value;
            return new EnumMemberValue(en.ordinal(), en.ordinal(), constPool);
        } else {
            MemberValue[] memberValues = castToMemberValues(value, constPool);
            ArrayMemberValue memberValue = new ArrayMemberValue(constPool);
            memberValue.setValue(memberValues);
            return memberValue;
        }
    }

    private static MemberValue[] castToMemberValues(Object value, ConstPool constPool) {
        Class<?> valueClass = value.getClass();
        MemberValue[] memberValues;
        if (valueClass.isArray()) {
            Object[] a = ReflectUtil.castAsArray(value);
            int len = a.length;
            memberValues = new MemberValue[len];
            for (int i = 0; i < len; i++) {
                Object v = a[i];
                memberValues[i] = castToMemberValue(v, constPool);
            }
        } else if (value instanceof Collection) {
            Collection<?> c = (Collection<?>) value;
            int len = c.size();
            int i = 0;
            memberValues = new MemberValue[len];
            for (Object v : c) {
                memberValues[i++] = castToMemberValue(v, constPool);
            }
        } else {
            throw new IllegalArgumentException(
                    "unsupported class " + valueClass.getCanonicalName());
        }
        return memberValues;
    }

    // ==== ==== ==== ====    ==== ==== ==== ====    ==== ==== ==== ====

    public static Class<?> toClass(String className) throws NotFoundException, CannotCompileException {
        return toCtClass(className).toClass();
    }

    public static CtClass[] toCtClass(Class<?>... classes) throws NotFoundException {
        int length = classes.length;
        CtClass[] ccs = new CtClass[length];
        for (int i = 0; i < length; i++) {
            ccs[i] = toCtClass(classes[i]);
        }
        return ccs;
    }

    public static CtClass toCtClass(Class<?> clazz) throws NotFoundException {
        return toCtClass(clazz.getName());
    }

    public static CtClass toCtClass(String className) throws NotFoundException {
        return classPool.get(className);
    }

    // ==== ==== ==== ====    ==== ==== ==== ====    ==== ==== ==== ====

    private static final ClassPool classPool = ClassPool.getDefault();

}
