package org.dreamcat.common.x.asm;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import javassist.CannotCompileException;
import javassist.ClassPool;
import javassist.CtClass;
import javassist.CtConstructor;
import javassist.CtField;
import javassist.CtMethod;
import javassist.NotFoundException;
import javassist.bytecode.AnnotationsAttribute;
import javassist.bytecode.ClassFile;

/**
 * Create by tuke on 2020/5/28
 */
public class MakeClass {

    private final ClassPool classPool;
    private final CtClass cc;
    private final ClassFile classFile;

    public MakeClass(String className) {
        this.classPool = ClassPool.getDefault();
        this.cc = classPool.makeClass(className);
        this.classFile = cc.getClassFile();
    }

    public static MakeClass make(String className) {
        return new MakeClass(className);
    }

    public Class<?> toClass() throws CannotCompileException {
        return toCtClass().toClass();
    }

    public CtClass toCtClass() {
        return cc;
    }

    // ==== ==== ==== ====    ==== ==== ==== ====    ==== ==== ==== ====

    public MakeClass superClass(Class<?> superClass)
            throws NotFoundException, CannotCompileException {
        return superClass(classPool.get(superClass.getCanonicalName()));
    }

    public MakeClass superClass(MakeClass superClass) throws CannotCompileException {
        return superClass(superClass.toCtClass());
    }

    public MakeClass superClass(CtClass superClass) throws CannotCompileException {
        cc.setSuperclass(superClass);
        return this;
    }

    // ---- ---- ---- ----    ---- ---- ---- ----    ---- ---- ---- ----

    public MakeClass interfaces(Collection<Class<?>> interfaces) throws NotFoundException {
        for (Class<?> i : interfaces) {
            cc.addInterface(CtClassUtil.toCtClass(i));
        }
        return this;
    }

    public MakeClass interfaceOf(Collection<MakeClass> interfaces) {
        for (MakeClass i : interfaces) {
            cc.addInterface(i.toCtClass());
        }
        return this;
    }

    public MakeClass interfaceFrom(Collection<CtClass> interfaces) {
        for (CtClass i : interfaces) {
            cc.addInterface(i);
        }
        return this;
    }

    public MakeClass addInterfaces(Class<?>... interfaces) throws NotFoundException {
        return interfaces(Arrays.asList(interfaces));
    }

    public MakeClass addInterfaceOf(MakeClass... interfaces) {
        return interfaceOf(Arrays.asList(interfaces));
    }

    public MakeClass addInterfaceFrom(CtClass... interfaces) {
        return interfaceFrom(Arrays.asList(interfaces));
    }

    // ---- ---- ---- ----    ---- ---- ---- ----    ---- ---- ---- ----

    public MakeClass annotation(String annotationClassName,
            String annotationMethod, String annotationMethodValue) {
        return annotation(annotationClassName, Collections.singletonMap(
                annotationMethod, annotationMethodValue));
    }

    public MakeClass annotation(String annotationClassName,
            Map<String, Object> annotationMembers) {
        return annotations(Collections.singletonMap(annotationClassName, annotationMembers));
    }

    public MakeClass annotations(Map<String, Map<String, Object>> annotations) {
        return annotations(CtClassUtil.createAnnotationsAttribute(
                classFile.getConstPool(), annotations));
    }

    public MakeClass annotations(AnnotationsAttribute annotationsAttribute) {
        classFile.addAttribute(annotationsAttribute);
        return this;
    }

    // ---- ---- ---- ----    ---- ---- ---- ----    ---- ---- ---- ----

    public MakeClass fields(Collection<MakeField> fields) throws CannotCompileException {
        for (MakeField i : fields) {
            cc.addField(i.toCtField());
            i.afterAdd();
        }
        return this;
    }

    public MakeClass fieldFrom(Collection<CtField> fields) throws CannotCompileException {
        for (CtField i : fields) {
            cc.addField(i);
        }
        return this;
    }

    public MakeClass fields(MakeField... fields) throws CannotCompileException {
        return fields(Arrays.asList(fields));
    }

    public MakeClass fieldFrom(CtField... fields) throws CannotCompileException {
        return fieldFrom(Arrays.asList(fields));
    }

    // ---- ---- ---- ----    ---- ---- ---- ----    ---- ---- ---- ----

    public MakeClass methods(Collection<MakeMethod> methods) throws CannotCompileException {
        for (MakeMethod i : methods) {
            cc.addMethod(i.toCtMethod());
            i.afterAdd();
        }
        return this;
    }

    public MakeClass methodFrom(Collection<CtMethod> methods) throws CannotCompileException {
        for (CtMethod i : methods) {
            cc.addMethod(i);
        }
        return this;
    }

    public MakeClass methods(MakeMethod... methods) throws CannotCompileException {
        return methods(Arrays.asList(methods));
    }

    public MakeClass methodFrom(CtMethod... methods) throws CannotCompileException {
        return methodFrom(Arrays.asList(methods));
    }

    // ---- ---- ---- ----    ---- ---- ---- ----    ---- ---- ---- ----

    public MakeClass constructors(Collection<MakeConstructor> constructors)
            throws CannotCompileException {
        for (MakeConstructor i : constructors) {
            cc.addConstructor(i.toCtConstructor());
            i.afterAdd();
        }
        return this;
    }

    public MakeClass constructorFrom(Collection<CtConstructor> constructors)
            throws CannotCompileException {
        for (CtConstructor i : constructors) {
            cc.addConstructor(i);
        }
        return this;
    }

    public MakeClass constructors(MakeConstructor... constructors) throws CannotCompileException {
        return constructors(Arrays.asList(constructors));
    }

    public MakeClass constructorFrom(CtConstructor... constructors) throws CannotCompileException {
        return constructorFrom(Arrays.asList(constructors));
    }
}
