package org.dreamcat.common.x.jackson;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.Writer;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import lombok.extern.slf4j.Slf4j;

/**
 * Create by tuke on 2018/10/19
 */
@Slf4j
@SuppressWarnings({"unchecked"})
public final class JsonUtil {

    private JsonUtil() {
    }

    private static final ObjectMapper objectMapper = new ObjectMapper();

    static {
        // json
        objectMapper.disable(SerializationFeature.FAIL_ON_EMPTY_BEANS);
    }

    public static void findAndRegisterModules() {
        registerModules(ObjectMapper.findModules());
    }

    public static void registerModules(Iterable<? extends Module> modules) {
        objectMapper.registerModules(modules);
    }

    // ==== ==== ==== ====    ==== ==== ==== ====    ==== ==== ==== ====

    public static <T> Map<String, Object> toMap(T bean) {
        return objectMapper.convertValue(bean,
                new TypeReference<Map<String, Object>>() {
                });
    }

    public static <T> T fromMap(Map<?, ?> map, Class<?> genericClass, Class<?>... parameterTypes) {
        return objectMapper.convertValue(map, getGenericType(genericClass, parameterTypes));
    }

    // ==== ==== ==== ====    ==== ==== ==== ====    ==== ==== ==== ====

    public static <T> T fromJson(String json, Class<T> clazz) {
        try {
            return objectMapper.readValue(json, clazz);
        } catch (IOException e) {
            log.error(e.getMessage());
            return null;
        }
    }

    public static <T> T fromJson(File file, Class<T> clazz) {
        try {
            return objectMapper.readValue(file, clazz);
        } catch (IOException e) {
            log.error(e.getMessage());
            return null;
        }
    }

    public static <T> T fromJson(Reader reader, Class<T> clazz) {
        try {
            return objectMapper.readValue(reader, clazz);
        } catch (IOException e) {
            log.error(e.getMessage());
            return null;
        }
    }

    public static <T> T fromJson(InputStream inputStream, Class<T> clazz) {
        try {
            return objectMapper.readValue(inputStream, clazz);
        } catch (IOException e) {
            log.error(e.getMessage());
            return null;
        }
    }

    // ---- ---- ---- ----    ---- ---- ---- ----    ---- ---- ---- ----

    /**
     * read as {@link LinkedHashMap}
     *
     * @param json json string
     * @return map object
     */
    public static Map<String, Object> fromJsonObject(String json) {
        return fromJson(json, Map.class);
    }

    public static <V> Map<String, V> fromJsonObject(String json, Class<V> valueType) {
        try {
            JavaType javaType = getGenericType(Map.class, String.class, valueType);
            return objectMapper.readValue(json, javaType);
        } catch (IOException e) {
            log.error(e.getMessage());
            return null;
        }
    }

    public static Map<String, Object> fromJsonObject(File file) {
        return fromJson(file, Map.class);
    }

    public static <V> Map<String, V> fromJsonObject(File file, Class<V> valueType) {
        try {
            JavaType javaType = getGenericType(Map.class, String.class, valueType);
            return objectMapper.readValue(file, javaType);
        } catch (IOException e) {
            log.error(e.getMessage());
            return null;
        }
    }

    public static Map<String, Object> fromJsonObject(Reader reader) {
        return fromJson(reader, Map.class);
    }

    public static <V> Map<String, V> fromJsonObject(Reader reader, Class<V> valueType) {
        try {
            JavaType javaType = getGenericType(Map.class, String.class, valueType);
            return objectMapper.readValue(reader, javaType);
        } catch (IOException e) {
            log.error(e.getMessage());
            return null;
        }
    }

    public static <V> Map<String, V> fromJsonObject(InputStream inputStream, Class<V> valueType) {
        try {
            JavaType javaType = getGenericType(Map.class, String.class, valueType);
            return objectMapper.readValue(inputStream, javaType);
        } catch (IOException e) {
            log.error(e.getMessage());
            return null;
        }
    }

    // ---- ---- ---- ----    ---- ---- ---- ----    ---- ---- ---- ----

    /**
     * read as {@link ArrayList}
     *
     * @param json json string
     * @return list object
     */
    public static <T> List<T> fromJsonArray(String json, Class<T> clazz) {
        try {
            return objectMapper.readValue(json, getGenericType(ArrayList.class, clazz));
        } catch (IOException e) {
            log.error(e.getMessage());
            return null;
        }
    }

    public static <T> List<T> fromJsonArray(File file, Class<T> clazz) {
        try {
            return objectMapper.readValue(file, getGenericType(ArrayList.class, clazz));
        } catch (IOException e) {
            log.error(e.getMessage());
            return null;
        }
    }

    public static <T> List<T> fromJsonArray(Reader reader, Class<T> clazz) {
        try {
            return objectMapper.readValue(reader, getGenericType(ArrayList.class, clazz));
        } catch (IOException e) {
            log.error(e.getMessage());
            return null;
        }
    }

    public static <T> List<T> fromJsonArray(InputStream inputStream, Class<T> clazz) {
        try {
            return objectMapper.readValue(inputStream, getGenericType(ArrayList.class, clazz));
        } catch (IOException e) {
            log.error(e.getMessage());
            return null;
        }
    }

    // ==== ==== ==== ====    ==== ==== ==== ====    ==== ==== ==== ====

    public static String toJson(Object bean) {
        try {
            return objectMapper.writeValueAsString(bean);
        } catch (JsonProcessingException e) {
            log.error(e.getMessage());
            return null;
        }
    }

    public static byte[] toJsonBytes(Object bean) {
        try {
            return objectMapper.writeValueAsBytes(bean);
        } catch (JsonProcessingException e) {
            log.error(e.getMessage());
            return null;
        }
    }

    // ---- ---- ---- ----    ---- ---- ---- ----    ---- ---- ---- ----

    public static boolean writeJson(Object bean, File file) {
        try {
            objectMapper.writeValue(file, bean);
            return true;
        } catch (IOException e) {
            log.error(e.getMessage());
            return false;
        }
    }

    public static boolean writeJson(Object bean, Writer writer) {
        try {
            objectMapper.writeValue(writer, bean);
            return true;
        } catch (IOException e) {
            log.error(e.getMessage());
            return false;
        }
    }

    /**
     * @param bean         java bean
     * @param outputStream output stream to save json
     * @return {@link false} if any error occurred
     * @see com.fasterxml.jackson.core.JsonEncoding#UTF8
     */
    public static boolean writeJson(Object bean, OutputStream outputStream) {
        try {
            objectMapper.writeValue(outputStream, bean);
            return true;
        } catch (IOException e) {
            log.error(e.getMessage());
            return false;
        }
    }

    // ==== ==== ==== ====    ==== ==== ==== ====    ==== ==== ==== ====

    public static JsonNode toJsonTree(String json) {
        try {
            return objectMapper.readTree(json);
        } catch (JsonProcessingException e) {
            log.error(e.getMessage());
            return null;
        }
    }

    public static Integer getInt(String json, String memberName) {
        return getInt(toJsonTree(json), memberName);
    }

    public static Integer getInt(JsonNode tree, String memberName) {
        if (tree != null && tree.isObject()) {
            JsonNode member = tree.get(memberName);
            if (member.isNumber()) {
                return member.asInt();
            }
        }
        return null;
    }

    public static Long getLong(String json, String memberName) {
        return getLong(toJsonTree(json), memberName);
    }

    public static Long getLong(JsonNode tree, String memberName) {
        if (tree != null && tree.isObject()) {
            JsonNode member = tree.get(memberName);
            if (member.isNumber()) {
                return member.asLong();
            }
        }
        return null;
    }

    public static Double getDouble(String json, String memberName) {
        return getDouble(toJsonTree(json), memberName);
    }

    public static Double getDouble(JsonNode tree, String memberName) {
        if (tree != null && tree.isObject()) {
            JsonNode member = tree.get(memberName);
            if (member.isNumber()) {
                return member.asDouble();
            }
        }
        return null;
    }

    public static String getString(String json, String memberName) {
        return getString(toJsonTree(json), memberName);
    }

    public static String getString(JsonNode tree, String memberName) {
        if (tree != null && tree.isObject()) {
            JsonNode member = tree.get(memberName);
            if (member.isNumber()) {
                return member.asText();
            }
        }
        return null;
    }

    public static Boolean getBoolean(String json, String memberName) {
        return getBoolean(toJsonTree(json), memberName);
    }

    public static Boolean getBoolean(JsonNode tree, String memberName) {
        if (tree != null && tree.isObject()) {
            JsonNode member = tree.get(memberName);
            if (member.isNumber()) {
                return member.asBoolean();
            }
        }
        return null;
    }

    // ==== ==== ==== ====    ==== ==== ==== ====    ==== ==== ==== ====

    public static JavaType getGenericType(Class<?> genericClass, Class<?>... parameterTypes) {
        return objectMapper.getTypeFactory().constructParametricType(genericClass, parameterTypes);
    }

}
