/*
 * Copyright (c) 2013-2024 Hutool Team and hutool.cn
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.dromara.hutool.db;

import org.dromara.hutool.core.lang.range.Segment;
import org.dromara.hutool.core.lang.page.PageInfo;
import org.dromara.hutool.core.array.ArrayUtil;
import org.dromara.hutool.db.sql.Order;

import java.io.Serializable;
import java.util.Arrays;

/**
 * 分页对象
 *
 * @author Looly
 */
public class Page implements Segment<Integer>, Serializable {
	private static final long serialVersionUID = 97792549823353462L;

	/**
	 * 默认
	 */
	public static final int DEFAULT_PAGE_SIZE = 20;

	/**
	 * 页码，0表示第一页
	 */
	private int pageNumber;
	/**
	 * 每页结果数
	 */
	private int pageSize;
	/**
	 * 排序
	 */
	private Order[] orders;

	/**
	 * 创建Page对象
	 *
	 * @param pageNumber 页码，0表示第一页
	 * @param pageSize   每页结果数
	 * @return Page
	 * @since 5.5.3
	 */
	public static Page of(final int pageNumber, final int pageSize) {
		return new Page(pageNumber, pageSize);
	}

	// ---------------------------------------------------------- Constructor start

	/**
	 * 构造，默认第0页，每页{@value #DEFAULT_PAGE_SIZE} 条
	 *
	 * @since 4.5.16
	 */
	public Page() {
		this(0, DEFAULT_PAGE_SIZE);
	}

	/**
	 * 构造
	 *
	 * @param pageNumber 页码，0表示第一页
	 * @param pageSize   每页结果数
	 */
	public Page(final int pageNumber, final int pageSize) {
		this.pageNumber = Math.max(pageNumber, 0);
		this.pageSize = pageSize <= 0 ? DEFAULT_PAGE_SIZE : pageSize;
	}

	/**
	 * 构造
	 *
	 * @param pageNumber 页码，0表示第一页
	 * @param pageSize   每页结果数
	 * @param order      排序对象
	 */
	public Page(final int pageNumber, final int pageSize, final Order order) {
		this(pageNumber, pageSize);
		this.orders = new Order[]{order};
	}
	// ---------------------------------------------------------- Constructor start

	// ---------------------------------------------------------- Getters and Setters start

	/**
	 * @return 页码，0表示第一页
	 */
	public int getPageNumber() {
		return pageNumber;
	}

	/**
	 * 设置页码，0表示第一页
	 *
	 * @param pageNumber 页码
	 */
	public void setPageNumber(final int pageNumber) {
		this.pageNumber = Math.max(pageNumber, 0);
	}

	/**
	 * @return 每页结果数
	 */
	public int getPageSize() {
		return pageSize;
	}

	/**
	 * 设置每页结果数
	 *
	 * @param pageSize 每页结果数
	 */
	public void setPageSize(final int pageSize) {
		this.pageSize = (pageSize <= 0) ? DEFAULT_PAGE_SIZE : pageSize;
	}

	/**
	 * @return 排序
	 */
	public Order[] getOrders() {
		return this.orders;
	}

	/**
	 * 设置排序
	 *
	 * @param orders 排序
	 */
	public void setOrder(final Order... orders) {
		this.orders = orders;
	}

	/**
	 * 设置排序
	 *
	 * @param orders 排序
	 * @return this
	 */
	public Page addOrder(final Order... orders) {
		this.orders = ArrayUtil.append(this.orders, orders);
		return this;
	}
	// ---------------------------------------------------------- Getters and Setters end

	@Override
	public Integer getBeginIndex() {
		return PageInfo.of(Integer.MAX_VALUE, this.pageSize)
				.setFirstPageNo(0).setPageNo(this.pageNumber).getBeginIndex();
	}

	@Override
	public Integer getEndIndex() {
		return PageInfo.of(Integer.MAX_VALUE, this.pageSize).setFirstPageNo(0).getEndIndex();
	}

	/**
	 * 开始位置和结束位置<br>
	 * 例如：
	 *
	 * <pre>
	 * 页码：0，每页10 =》 [0, 10]
	 * 页码：1，每页10 =》 [10, 20]
	 * 页码：2，每页10 =》 [21, 30]
	 * 。。。
	 * </pre>
	 *
	 * @return 第一个数为开始位置，第二个数为结束位置
	 */
	public int[] getStartEnd() {
		final PageInfo pageInfo = PageInfo.of(Integer.MAX_VALUE, this.pageSize)
				.setFirstPageNo(0).setPageNo(this.pageNumber);
		return new int[]{pageInfo.getBeginIndex(), pageInfo.getEndIndexExclude()};
	}

	@Override
	public String toString() {
		return "Page [page=" + pageNumber + ", pageSize=" + pageSize + ", order=" + Arrays.toString(orders) + "]";
	}
}
