/*
 * Copyright (c) 2023 looly(loolly@aliyun.com)
 * Hutool is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package org.dromara.hutool;

import org.dromara.hutool.core.lang.ConsoleTable;
import org.dromara.hutool.core.reflect.ClassUtil;
import org.dromara.hutool.core.text.StrUtil;

import java.util.Set;

/**
 * <pre>
 * ========================================
 *     __  __        __                   __
 *    / / / /__  __ / /_ __ __   __ __   / /
 *   / /_/ // / / // __// _ _ \ / _ _ \ / /
 *  / __  // /_/ // /_ / /__/ // /__/ // /
 * /_/ /_/ \____/ \__/ \_____/ \_____//_/
 *
 * -----------https://hutool.cn/-----------
 * ========================================
 * </pre>
 *
 * <p>
 * Hutool是一个小而全的Java工具类库，通过静态方法封装，降低相关API的学习成本，提高工作效率，使Java拥有函数式语言般的优雅，让Java语言也可以“甜甜的”。
 * </p>
 *
 * <p>
 * Hutool中的工具方法来自于每个用户的精雕细琢，它涵盖了Java开发底层代码中的方方面面，它既是大型项目开发中解决小问题的利器，也是小型项目中的效率担当；<br>
 * </p>
 *
 * <p>Hutool是项目中“util”包友好的替代，它节省了开发人员对项目中公用类和公用工具方法的封装时间，使开发专注于业务，同时可以最大限度的避免封装不完善带来的bug。</p>
 *
 * @author Looly
 */
public class Hutool {

	/**
	 * 作者（贡献者）
	 */
	public static final String AUTHOR = "Looly";

	private Hutool() {
	}

	/**
	 * 显示Hutool所有的工具类
	 *
	 * @return 工具类名集合
	 * @since 5.5.2
	 */
	public static Set<Class<?>> getAllUtils() {
		return ClassUtil.scanPackage("org.dromara.hutool",
				(clazz) -> (!clazz.isInterface()) && StrUtil.endWith(clazz.getSimpleName(), "Util"));
	}

	/**
	 * 控制台打印所有工具类
	 */
	public static void printAllUtils() {
		final Set<Class<?>> allUtils = getAllUtils();
		final ConsoleTable consoleTable = ConsoleTable.of().addHeader("工具类名", "所在包");
		for (final Class<?> clazz : allUtils) {
			consoleTable.addBody(clazz.getSimpleName(), clazz.getPackage().getName());
		}
		consoleTable.print();
	}
}
