/*
 * Decompiled with CFR 0.152.
 */
package org.dromara.mica.mqtt.core.client;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.dromara.mica.mqtt.codec.MqttMessage;
import org.dromara.mica.mqtt.codec.MqttMessageBuilders;
import org.dromara.mica.mqtt.codec.MqttProperties;
import org.dromara.mica.mqtt.codec.MqttPublishMessage;
import org.dromara.mica.mqtt.codec.MqttQoS;
import org.dromara.mica.mqtt.codec.MqttSubscribeMessage;
import org.dromara.mica.mqtt.codec.MqttUnsubscribeMessage;
import org.dromara.mica.mqtt.core.client.IMqttClient;
import org.dromara.mica.mqtt.core.client.IMqttClientMessageListener;
import org.dromara.mica.mqtt.core.client.IMqttClientSession;
import org.dromara.mica.mqtt.core.client.MqttClientCreator;
import org.dromara.mica.mqtt.core.client.MqttClientSubscription;
import org.dromara.mica.mqtt.core.client.MqttPendingSubscription;
import org.dromara.mica.mqtt.core.client.MqttPendingUnSubscription;
import org.dromara.mica.mqtt.core.common.MqttPendingPublish;
import org.dromara.mica.mqtt.core.serializer.MqttSerializer;
import org.dromara.mica.mqtt.core.util.TopicUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tio.client.ClientChannelContext;
import org.tio.client.TioClient;
import org.tio.client.TioClientConfig;
import org.tio.core.ChannelContext;
import org.tio.core.Node;
import org.tio.core.Tio;
import org.tio.core.intf.Packet;
import org.tio.utils.timer.TimerTask;
import org.tio.utils.timer.TimerTaskService;

public final class MqttClient
implements IMqttClient {
    private static final Logger logger = LoggerFactory.getLogger(MqttClient.class);
    private final TioClient tioClient;
    private final MqttClientCreator config;
    private final TioClientConfig clientTioConfig;
    private final IMqttClientSession clientSession;
    private final TimerTaskService taskService;
    private final ExecutorService mqttExecutor;
    private final MqttSerializer mqttSerializer;
    private ClientChannelContext context;

    public static MqttClientCreator create() {
        return new MqttClientCreator();
    }

    MqttClient(TioClient tioClient, MqttClientCreator config) {
        this.tioClient = tioClient;
        this.config = config;
        this.clientTioConfig = tioClient.getClientConfig();
        this.taskService = config.getTaskService();
        this.mqttExecutor = config.getMqttExecutor();
        this.clientSession = config.getClientSession();
        this.mqttSerializer = config.getMqttSerializer();
    }

    public MqttClient subQos0(String topicFilter, IMqttClientMessageListener listener) {
        return this.subscribe(topicFilter, MqttQoS.QOS0, listener);
    }

    public MqttClient subQos1(String topicFilter, IMqttClientMessageListener listener) {
        return this.subscribe(topicFilter, MqttQoS.QOS1, listener);
    }

    public MqttClient subQos2(String topicFilter, IMqttClientMessageListener listener) {
        return this.subscribe(topicFilter, MqttQoS.QOS2, listener);
    }

    public MqttClient subscribe(MqttQoS mqttQoS, String topicFilter, IMqttClientMessageListener listener) {
        return this.subscribe(topicFilter, mqttQoS, listener, null);
    }

    public MqttClient subscribe(String topicFilter, MqttQoS mqttQoS, IMqttClientMessageListener listener) {
        return this.subscribe(topicFilter, mqttQoS, listener, null);
    }

    public MqttClient subscribe(String topicFilter, MqttQoS mqttQoS, IMqttClientMessageListener listener, MqttProperties properties) {
        return this.subscribe(Collections.singletonList(new MqttClientSubscription(mqttQoS, topicFilter, listener)), properties);
    }

    public MqttClient subscribe(String[] topicFilters, MqttQoS mqttQoS, IMqttClientMessageListener listener) {
        return this.subscribe(topicFilters, mqttQoS, listener, null);
    }

    public MqttClient subscribe(String[] topicFilters, MqttQoS mqttQoS, IMqttClientMessageListener listener, MqttProperties properties) {
        Objects.requireNonNull(topicFilters, "MQTT subscribe topicFilters is null.");
        ArrayList<MqttClientSubscription> subscriptionList = new ArrayList<MqttClientSubscription>();
        for (String topicFilter : topicFilters) {
            subscriptionList.add(new MqttClientSubscription(mqttQoS, topicFilter, listener));
        }
        return this.subscribe(subscriptionList, properties);
    }

    public MqttClient subscribe(List<MqttClientSubscription> subscriptionList) {
        return this.subscribe(subscriptionList, null);
    }

    public MqttClient subscribe(List<MqttClientSubscription> subscriptionList, MqttProperties properties) {
        ArrayList<MqttClientSubscription> needSubscriptionList = new ArrayList<MqttClientSubscription>();
        for (MqttClientSubscription subscription : subscriptionList) {
            TopicUtil.validateTopicFilter((String)subscription.getTopicFilter());
            boolean subscribed = this.clientSession.isSubscribed(subscription);
            if (subscribed) continue;
            needSubscriptionList.add(subscription);
        }
        if (needSubscriptionList.isEmpty()) {
            return this;
        }
        List topicSubscriptionList = needSubscriptionList.stream().map(MqttClientSubscription::toTopicSubscription).collect(Collectors.toList());
        int messageId = this.clientSession.getPacketId();
        MqttSubscribeMessage message = MqttMessageBuilders.subscribe().addSubscriptions(topicSubscriptionList).messageId(messageId).properties(properties).build();
        ClientChannelContext clientContext = this.getContext();
        if (clientContext != null && clientContext.isAccepted()) {
            MqttPendingSubscription pendingSubscription = new MqttPendingSubscription(needSubscriptionList, message);
            pendingSubscription.startRetransmitTimer(this.taskService, (ChannelContext)clientContext);
            this.clientSession.addPaddingSubscribe(messageId, pendingSubscription);
            boolean result = Tio.send((ChannelContext)clientContext, (Packet)message);
            logger.info("MQTT subscriptionList:{} messageId:{} subscribing result:{}", new Object[]{needSubscriptionList, messageId, result});
        } else {
            this.clientSession.addSubscriptionList(needSubscriptionList);
        }
        return this;
    }

    public MqttClient unSubscribe(String ... topicFilters) {
        return this.unSubscribe(Arrays.asList(topicFilters));
    }

    public MqttClient unSubscribe(List<String> topicFilters) {
        TopicUtil.validateTopicFilter(topicFilters);
        this.clientSession.removePaddingSubscribes(topicFilters);
        this.clientSession.removeSubscriptions(topicFilters);
        int messageId = this.clientSession.getPacketId();
        MqttUnsubscribeMessage message = MqttMessageBuilders.unsubscribe().addTopicFilters(topicFilters).messageId(messageId).build();
        MqttPendingUnSubscription pendingUnSubscription = new MqttPendingUnSubscription(topicFilters, message);
        ClientChannelContext clientContext = this.getContext();
        this.clientSession.addPaddingUnSubscribe(messageId, pendingUnSubscription);
        pendingUnSubscription.startRetransmissionTimer(this.taskService, (ChannelContext)clientContext);
        boolean result = Tio.send((ChannelContext)clientContext, (Packet)message);
        logger.info("MQTT Topic:{} messageId:{} unSubscribing result:{}", new Object[]{topicFilters, messageId, result});
        return this;
    }

    public boolean publish(String topic, Object payload) {
        return this.publish(topic, payload, MqttQoS.QOS0);
    }

    public boolean publish(String topic, Object payload, MqttQoS qos) {
        return this.publish(topic, payload, qos, false);
    }

    public boolean publish(String topic, Object payload, boolean retain) {
        return this.publish(topic, payload, MqttQoS.QOS0, retain);
    }

    public boolean publish(String topic, Object payload, MqttQoS qos, boolean retain) {
        return this.publish(topic, payload, qos, publishBuilder -> publishBuilder.retained(retain));
    }

    public boolean publish(String topic, Object payload, MqttQoS qos, boolean retain, MqttProperties properties) {
        return this.publish(topic, payload, qos, publishBuilder -> publishBuilder.retained(retain).properties(properties));
    }

    public boolean publish(String topic, Object payload, MqttQoS qos, Consumer<MqttMessageBuilders.PublishBuilder> builder) {
        TopicUtil.validateTopicName((String)topic);
        boolean isHighLevelQoS = MqttQoS.QOS1 == qos || MqttQoS.QOS2 == qos;
        int messageId = isHighLevelQoS ? this.clientSession.getPacketId() : -1;
        MqttMessageBuilders.PublishBuilder publishBuilder = MqttMessageBuilders.publish();
        byte[] newPayload = payload instanceof byte[] ? (byte[])payload : this.mqttSerializer.serialize(payload);
        builder.accept(publishBuilder);
        publishBuilder.topicName(topic).payload(newPayload).messageId(messageId).qos(qos);
        MqttPublishMessage message = publishBuilder.build();
        ClientChannelContext clientContext = this.getContext();
        if (clientContext == null) {
            logger.error("MQTT client publish fail, TCP not connected.");
            return false;
        }
        if (!clientContext.isAccepted()) {
            logger.error("TCP is connected but mqtt is not accepted.");
            return false;
        }
        if (isHighLevelQoS) {
            MqttPendingPublish pendingPublish = new MqttPendingPublish(message, qos);
            this.clientSession.addPendingPublish(messageId, pendingPublish);
            pendingPublish.startPublishRetransmissionTimer(this.taskService, (ChannelContext)clientContext);
        }
        boolean result = Tio.send((ChannelContext)clientContext, (Packet)message);
        logger.debug("MQTT Topic:{} qos:{} retain:{} publish result:{}", new Object[]{topic, qos, publishBuilder.isRetained(), result});
        return result;
    }

    public TimerTask schedule(Runnable command, long delay) {
        return this.tioClient.schedule(command, delay);
    }

    public TimerTask schedule(Runnable command, long delay, Executor executor) {
        return this.tioClient.schedule(command, delay, executor);
    }

    public TimerTask scheduleOnce(Runnable command, long delay) {
        return this.tioClient.scheduleOnce(command, delay);
    }

    public TimerTask scheduleOnce(Runnable command, long delay, Executor executor) {
        return this.tioClient.scheduleOnce(command, delay, executor);
    }

    MqttClient start(boolean sync) {
        Node node = new Node(this.config.getIp(), this.config.getPort());
        try {
            if (sync) {
                this.tioClient.connect(node, this.config.getBindIp(), Integer.valueOf(0), this.config.getTimeout());
            } else {
                this.tioClient.asyncConnect(node, this.config.getBindIp(), Integer.valueOf(0), this.config.getTimeout());
            }
            return this;
        }
        catch (Exception e) {
            throw new IllegalStateException("Mica mqtt client async start fail.", e);
        }
    }

    public void reconnect() {
        ClientChannelContext channelContext = this.getContext();
        if (channelContext == null) {
            return;
        }
        try {
            if (channelContext.isRemoved()) {
                channelContext.setRemoved(false);
            }
            this.tioClient.reconnect(channelContext, this.config.getTimeout());
        }
        catch (Exception e) {
            logger.error("mqtt client reconnect error", (Throwable)e);
        }
    }

    public boolean reconnect(String ip, int port) {
        return this.reconnect(new Node(ip, port));
    }

    public boolean reconnect(Node serverNode) {
        this.config.ip(serverNode.getIp()).port(serverNode.getPort());
        ClientChannelContext oldContext = this.getContext();
        if (oldContext == null) {
            Set closedSet = this.clientTioConfig.closeds;
            if (closedSet != null && !closedSet.isEmpty()) {
                ChannelContext closedContext = (ChannelContext)closedSet.iterator().next();
                closedContext.setServerNode(serverNode);
            }
        } else {
            oldContext.setServerNode(serverNode);
            Tio.close((ChannelContext)oldContext, (String)("\u5207\u6362\u670d\u52a1\u5730\u5740\uff1a" + serverNode));
        }
        return false;
    }

    public boolean disconnect() {
        ClientChannelContext channelContext = this.getContext();
        if (channelContext == null) {
            return false;
        }
        boolean result = Tio.bSend((ChannelContext)channelContext, (Packet)MqttMessage.DISCONNECT);
        if (result) {
            Tio.close((ChannelContext)channelContext, null, (String)"MqttClient disconnect.", (boolean)true);
        }
        return result;
    }

    public boolean stop() {
        this.disconnect();
        boolean result = this.tioClient.stop();
        try {
            this.mqttExecutor.shutdown();
        }
        catch (Exception e1) {
            logger.error(e1.getMessage(), (Throwable)e1);
        }
        try {
            result &= this.mqttExecutor.awaitTermination(6L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            logger.error(e.getMessage(), (Throwable)e);
        }
        logger.info("MqttClient stop result:{}", (Object)result);
        this.clientSession.clean();
        return result;
    }

    public TioClient getTioClient() {
        return this.tioClient;
    }

    public MqttClientCreator getClientCreator() {
        return this.config;
    }

    public TioClientConfig getClientTioConfig() {
        return this.clientTioConfig;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ClientChannelContext getContext() {
        if (this.context != null) {
            return this.context;
        }
        MqttClient mqttClient = this;
        synchronized (mqttClient) {
            Set contextSet;
            if (this.context == null && (contextSet = Tio.getConnecteds((TioClientConfig)this.clientTioConfig)) != null && !contextSet.isEmpty()) {
                this.context = (ClientChannelContext)contextSet.iterator().next();
            }
        }
        return this.context;
    }

    public boolean isConnected() {
        ClientChannelContext channelContext = this.getContext();
        return channelContext != null && channelContext.isAccepted();
    }

    public boolean isDisconnected() {
        return !this.isConnected();
    }

    @Override
    public MqttClient getMqttClient() {
        return this;
    }
}

