/*
 * Copyright (c) 2019-2029, Dreamlu 卢春梦 (596392912@qq.com & dreamlu.net).
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.dromara.mica.mqtt.codec.message.properties;

import org.dromara.mica.mqtt.codec.properties.*;

/**
 * mqtt5 连接属性
 *
 * @author L.cm
 */
public class MqttConnectProperties {
	private final MqttProperties properties;

	public MqttConnectProperties() {
		this(new MqttProperties());
	}

	public MqttConnectProperties(MqttProperties properties) {
		this.properties = properties;
	}

	public MqttProperties getProperties() {
		return properties;
	}

	/**
	 * 设置会话过期时间
	 *
	 * @param sessionExpiryInterval 会话过期时间
	 * @return MqttConnectProperty
	 */
	public MqttConnectProperties setSessionExpiryInterval(int sessionExpiryInterval) {
		properties.add(new IntegerProperty(MqttPropertyType.SESSION_EXPIRY_INTERVAL, sessionExpiryInterval));
		return this;
	}

	/**
	 * 设置认证方法
	 *
	 * @param authenticationMethod 认证方法
	 * @return MqttConnectProperty
	 */
	public MqttConnectProperties setAuthenticationMethod(String authenticationMethod) {
		properties.add(new StringProperty(MqttPropertyType.AUTHENTICATION_METHOD, authenticationMethod));
		return this;
	}

	/**
	 * 设置认证数据
	 *
	 * @param authenticationData 认证数据
	 * @return MqttConnectProperty
	 */
	public MqttConnectProperties setAuthenticationData(byte[] authenticationData) {
		properties.add(new BinaryProperty(MqttPropertyType.AUTHENTICATION_DATA, authenticationData));
		return this;
	}

	/**
	 * 设置请求问题信息
	 *
	 * @param requestProblemInformation 请求问题信息
	 * @return MqttConnectProperty
	 */
	public MqttConnectProperties setRequestProblemInformation(boolean requestProblemInformation) {
		properties.add(new BooleanProperty(MqttPropertyType.REQUEST_PROBLEM_INFORMATION, requestProblemInformation));
		return this;
	}

	/**
	 * 设置请求响应信息
	 *
	 * @param requestResponseInformation 请求响应信息
	 * @return MqttConnectProperty
	 */
	public MqttConnectProperties setRequestResponseInformation(boolean requestResponseInformation) {
		properties.add(new BooleanProperty(MqttPropertyType.REQUEST_RESPONSE_INFORMATION, requestResponseInformation));
		return this;
	}

	/**
	 * 设置接收最大包数
	 *
	 * @param receiveMaximum 接收最大包数
	 * @return MqttConnectProperty
	 */
	public MqttConnectProperties setReceiveMaximum(int receiveMaximum) {
		properties.add(new IntegerProperty(MqttPropertyType.RECEIVE_MAXIMUM, receiveMaximum));
		return this;
	}

	/**
	 * 设置主题别名最大数
	 *
	 * @param topicAliasMaximum 主题别名最大数
	 * @return MqttConnectProperty
	 */
	public MqttConnectProperties setTopicAliasMaximum(int topicAliasMaximum) {
		properties.add(new IntegerProperty(MqttPropertyType.TOPIC_ALIAS_MAXIMUM, topicAliasMaximum));
		return this;
	}

	/**
	 * 设置最大包大小
	 *
	 * @param maximumPacketSize 最大包大小
	 * @return MqttConnectProperty
	 */
	public MqttConnectProperties setMaximumPacketSize(int maximumPacketSize) {
		properties.add(new IntegerProperty(MqttPropertyType.MAXIMUM_PACKET_SIZE, maximumPacketSize));
		return this;
	}

	/**
	 * 设置用户属性
	 *
	 * @param userProperty 用户属性
	 * @return MqttConnectProperty
	 */
	public MqttConnectProperties addUserProperty(UserProperty userProperty) {
		properties.add(userProperty);
		return this;
	}

	/**
	 * 添加用户属性
	 *
	 * @param key   key
	 * @param value value
	 * @return MqttConnectProperty
	 */
	public MqttConnectProperties addUserProperty(String key, String value) {
		this.addUserProperty(new UserProperty(key, value));
		return this;
	}
}
