package org.dromara.northstar.strategy.model;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Predicate;

import org.apache.commons.lang3.StringUtils;
import org.dromara.northstar.common.TickDataAware;
import org.dromara.northstar.common.TransactionAware;
import org.dromara.northstar.common.constant.SignalOperation;
import org.dromara.northstar.common.utils.OrderUtils;
import org.dromara.northstar.strategy.IModuleContext;
import org.dromara.northstar.strategy.constant.PriceType;
import org.springframework.util.Assert;

import lombok.Builder;
import lombok.Getter;
import lombok.NonNull;
import lombok.Setter;
import xyz.redtorch.pb.CoreEnum.DirectionEnum;
import xyz.redtorch.pb.CoreField.ContractField;
import xyz.redtorch.pb.CoreField.OrderField;
import xyz.redtorch.pb.CoreField.TickField;
import xyz.redtorch.pb.CoreField.TradeField;


/**
 * 交易意图
 * 封装自动撤单追单逻辑
 * @author KevinHuangwl
 *
 */
public class TradeIntent implements TransactionAware, TickDataAware {
	
	@Setter
	private IModuleContext context;
	/**
	 * 合约
	 */
	@Getter
	@NonNull
	private final ContractField contract;
	/**
	 * 操作
	 */
	@Getter
	@NonNull
	private final SignalOperation operation;
	/**
	 * 价格类型
	 */
	@NonNull
	@Getter
	private final PriceType priceType;
	/**
	 * 价格
	 */
	@Getter
	private final double price;
	/**
	 * 目标手数
	 */
	@Getter
	private final int volume;
	/**
	 * 订单超时（毫秒）
	 */
	private final long timeout;
	/**
	 * 价差过大的放弃条件
	 */
	private final Predicate<Double> priceDiffConditionToAbort;
	/**
	 * 意图初始价
	 */
	private Double initialPrice;
	
	@Builder
	public TradeIntent(ContractField contract, SignalOperation operation, PriceType priceType, double price, int volume, 
			long timeout, Predicate<Double> priceDiffConditionToAbort) {
		Assert.noNullElements(List.of(contract, operation, priceType), "入参不能为空");
		Assert.isTrue(volume > 0, "手数必须为正整数");
		Assert.isTrue(timeout > 0, "订单等待时长必须为正整数");
		this.contract = contract;
		this.operation = operation;
		this.priceType = priceType;
		this.price = price;
		this.volume = volume;
		this.timeout = timeout;
		this.priceDiffConditionToAbort = priceDiffConditionToAbort;
	}
	
	private Optional<String> orderIdRef = Optional.empty();

	private int accVol;
	
	private boolean terminated;
	
	private long lastCancelReqTime;
	
	@Override
	public synchronized void onTick(TickField tick) {
		if(!StringUtils.equals(tick.getUnifiedSymbol(), contract.getUnifiedSymbol())) 
			return;
		
		if(Objects.isNull(initialPrice)) {
			initialPrice = tick.getLastPrice();
			context.getLogger().debug("交易意图初始价位：{}", initialPrice);
		}
		if(Objects.nonNull(priceDiffConditionToAbort)) {
			double priceDiff = Math.abs(tick.getLastPrice() - initialPrice);
			terminated = priceDiffConditionToAbort.test(priceDiff);
			if(terminated) {
				context.getLogger().info("{} {} 价差过大中止交易意图，当前价差为{}", tick.getActionDay(), tick.getActionTime(), priceDiff);
				orderIdRef.ifPresent(context::cancelOrder);
			}
		}
		if(hasTerminated()) {
			context.getLogger().debug("交易意图已终止");
			return;
		}
		if(orderIdRef.isEmpty() && !context.getState().isOrdering()) {
			context.getLogger().debug("交易意图自动发单");
			orderIdRef = context.submitOrderReq(contract, operation, priceType, orderVol(), price);
		} else if (orderIdRef.isPresent() && context.isOrderWaitTimeout(orderIdRef.get(), timeout) && tick.getActionTimestamp() - lastCancelReqTime > 3000) {
			context.getLogger().debug("交易意图自动撤单");
			context.cancelOrder(orderIdRef.get());
			lastCancelReqTime = tick.getActionTimestamp();
		}
	}
	
	private int orderVol() {
		// 开仓信号下，不需要分拆计算
		if(operation.isOpen()) {
			return restVol();
		}
		// 平仓信号下，要确保发单数量不超过今仓昨仓数量
		// 例如平仓数量为2，今仓为1，昨仓为1。就要分拆成先后两次下单
		// 分拆过程中还要考虑模组当前的平仓优化策略
		DirectionEnum direction = switch (operation) {
		case BUY_CLOSE -> DirectionEnum.D_Sell;
		case SELL_CLOSE -> DirectionEnum.D_Buy;
		default -> throw new IllegalArgumentException("Unexpected value: " + operation);
		};
		int tdVol = context.getModuleAccount().getNonclosedPosition(contract.getUnifiedSymbol(), direction, true);
		int ydVol = context.getModuleAccount().getNonclosedPosition(contract.getUnifiedSymbol(), direction, false);
		// 其中一个为零时，不需要分拆计算
		if(tdVol * ydVol == 0) {
			return restVol();
		}
		return switch(context.getModule().getModuleDescription().getClosingPolicy()) {
		case CLOSE_NONTODAY_HEGDE_TODAY -> ydVol > 0 ? Math.min(ydVol, restVol()) : restVol();
		case FIRST_IN_FIRST_OUT -> Math.min(ydVol, restVol());
		case FIRST_IN_LAST_OUT -> Math.min(tdVol, restVol()); 
		default -> throw new IllegalStateException("未定义类型：" + context.getModule().getModuleDescription().getClosingPolicy());
		};
	}
	
	private int restVol() {
		return volume - accVol;
	}

	@Override
	public synchronized void onOrder(OrderField order) {
		// 订单结束
		orderIdRef
			.filter(id -> StringUtils.equals(id, order.getOriginOrderId()))
			.ifPresent(id -> {
				if(OrderUtils.isDoneOrder(order)) {	
					orderIdRef = Optional.empty();
				}
			});
	}

	@Override
	public synchronized void onTrade(TradeField trade) {
		accVol += trade.getVolume();
	}

	public boolean hasTerminated() {
		return terminated || accVol == volume;
	}
	
	@Override
	public String toString() {
		return String.format("TradeIntent [contract=%s, operation=%s, priceType=%s, price=%s, volume=%s, timeout=%s]", 
				contract.getContractId(), operation, priceType, price, volume, timeout);
	}
	
}
