package ord.dronda.scheduling.cronk

import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.Job
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.delay
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import ord.dronda.scheduling.cronk.lib.secondsUntilNextExecution
import org.slf4j.LoggerFactory

public class CronkScheduler(dispatcher: CoroutineDispatcher) {
    private val supervisor = SupervisorJob()
    private val scope = CoroutineScope(dispatcher + supervisor)
    private val jobs = mutableMapOf<String, ScheduledJob>()
    private val scheduleLock = Mutex()
    private val logger = LoggerFactory.getLogger(this::class.java)

    public suspend fun schedule(cronJob: CronJob): Boolean = scheduleLock.withLock {
        if (jobs.containsKey(cronJob.id)) {
            return@withLock false
        }

        val determineNextExecution: () -> Long = {
            cronJob.cron.secondsUntilNextExecution(cronJob.zoneId).toMilliseconds()
        }

        val job = scope.launch {
            while (isActive) {
                delay(determineNextExecution())
                try {
                    cronJob.job()
                } catch (e: Exception) {
                    logger.error("Job ${cronJob.name}, failed", e)
                }
            }
        }
        jobs[cronJob.id] = ScheduledJob(job, cronJob)

        true
    }

    public fun removeJob(id: String) {
        val job = jobs[id] ?: return
        job.coroutineJob.cancel()
        jobs.remove(id)
    }

    public data class ScheduledJob(val coroutineJob: Job, val cronJob: CronJob)

    public companion object {
        @OptIn(ExperimentalCoroutinesApi::class)
        public fun default(): CronkScheduler = CronkScheduler(Dispatchers.IO.limitedParallelism(2))
        private fun Long.toMilliseconds() = this * 1000
    }
}