package org.dronda.lib.aggregate.processor

import org.dronda.lib.aggregate.Aggregate
import org.dronda.lib.aggregate.Event
import org.dronda.lib.aggregate.ValidatingAggregator
import org.dronda.lib.aggregate.reactor.EventReactor

internal typealias EnrichFromDataStore<A> = suspend (modelId: String) -> A?
internal typealias EventApplier<E, A> = (event: E, aggregate: A?) -> ValidatingAggregator.Result<A>
internal typealias UpdateDataStore<A> = suspend (modelId: String, aggregate: A) -> DataStoreResult
internal typealias ReactToUpdatedAggregate<E, A> = (AggregateProcessorEnvelope<E, A>) -> Unit

public class AggregateProcessor<E : Event, A : Aggregate> private constructor(
    private val loader: EnrichFromDataStore<A>,
    private val applier: EventApplier<E, A>,
    private val updater: UpdateDataStore<A>,
    private val reactor: ReactToUpdatedAggregate<E, A>,
    private val maxRetries: Int
): Processor<E, A> {
    private companion object {
        private const val MAX_RETRIES = 10
    }
    public constructor(
        dataStore: DataStore<A>,
        aggregator: ValidatingAggregator<E, A>,
        eventReactors: List<EventReactor<E, A>> = emptyList(),
    ) : this(
        DataStoreEnricher<E, A>(dataStore).impl,
        AggregateApplier<E, A>(aggregator).impl,
        DataStoreUpdater<A>(dataStore).impl,
        ReactorApplier<E, A>(eventReactors).impl,
        maxRetries = MAX_RETRIES
    )
    override suspend fun process(event: E): ProcessorResult {
        val loadResult = loader(event.modelId)
        val aggregate = when (val result = applier(event, loadResult)) {
            is ValidatingAggregator.Result.Invalid -> return ProcessorResult.Failure
            is ValidatingAggregator.Result.Valid -> result.aggregate
        }

        for(i in 0 until maxRetries) {
            when (updater(event.modelId, aggregate)) {
                is DataStoreResult.Success -> break
                is DataStoreResult.Failure.Retry -> continue
                else -> return ProcessorResult.Failure
            }
        }
        reactor(AggregateProcessorEnvelope(event, aggregate))
        return ProcessorResult.Success
    }
}

