package org.dronda.lib.jvm.encryption

import java.io.InputStream
import java.util.Base64
import javax.crypto.Cipher
import javax.crypto.CipherInputStream
import javax.crypto.SecretKey
import javax.crypto.spec.IvParameterSpec
import javax.crypto.spec.SecretKeySpec

public class AesCbcPkcs5Encryptor(
    private val secretKey: SecretKey
) : Encryptor {

    public companion object {
        public fun fromBase64(secretKey: String): AesCbcPkcs5Encryptor {
            val key = Base64.getDecoder().decode(secretKey)
            return AesCbcPkcs5Encryptor(
                SecretKeySpec(key, AlgorithmUtil.AES)
            )
        }
        public fun from(secretKey: ByteArray): AesCbcPkcs5Encryptor {
            return AesCbcPkcs5Encryptor(
                SecretKeySpec(secretKey, AlgorithmUtil.AES)
            )
        }
        public fun getExpectedPaddedSize(size: Long): Long {
            val modded = size % 16
            return size + (16 - modded)
        }
    }
    public override fun encryptPayload(payload: ByteArray): AESEncryptionData {
        val iv = createInitializationVector(16)
        val encryptedValue = performAESEncryption(payload, secretKey, iv)
        return AESEncryptionData(encryptedValue, iv)
    }

    public override fun encryptPayload(payload: InputStream): AESInputStreamData {
        val iv = createInitializationVector(16)
        val encryptionStream = performAESEncryption(payload, secretKey, iv)
        return AESInputStreamData(
            value = encryptionStream,
            iv = iv,
        )
    }

    public override fun decryptPayload(encryption: AESEncryptionData): ByteArray {
        return performAESDecryption(encryption.value, secretKey, encryption.iv)
    }

    public override fun decryptPayload(encryption: AESInputStreamData): InputStream {
        return performAESDecryption(encryption.value, secretKey, encryption.iv)
    }


    private fun performAESEncryption(payload: ByteArray, secretKey: SecretKey, iv: ByteArray): ByteArray {
        val cipher = Cipher.getInstance(AlgorithmUtil.AES_CBC_PKCS5_ALGORITHM)
        val ivParameterSpec = IvParameterSpec(iv)
        cipher.init(Cipher.ENCRYPT_MODE, secretKey, ivParameterSpec)
        return cipher.doFinal(payload)
    }

    private fun performAESEncryption(payload: InputStream, secretKey: SecretKey, iv: ByteArray): InputStream {
        val cipher = Cipher.getInstance(AlgorithmUtil.AES_CBC_PKCS5_ALGORITHM)
        val ivParameterSpec = IvParameterSpec(iv)
        cipher.init(Cipher.ENCRYPT_MODE, secretKey, ivParameterSpec)
        return CipherInputStream(payload, cipher)
    }

    private fun performAESDecryption(cipherText: ByteArray, secretKey: SecretKey, iv: ByteArray) : ByteArray {
        val cipher = Cipher.getInstance(AlgorithmUtil.AES_CBC_PKCS5_ALGORITHM)
        val ivParameterSpec = IvParameterSpec(iv)
        cipher.init(Cipher.DECRYPT_MODE, secretKey, ivParameterSpec)
        return cipher.doFinal(cipherText)
    }

    private fun performAESDecryption(payload: InputStream, secretKey: SecretKey, iv: ByteArray): InputStream {
        val cipher = Cipher.getInstance(AlgorithmUtil.AES_CBC_PKCS5_ALGORITHM)
        val ivParameterSpec = IvParameterSpec(iv)
        cipher.init(Cipher.DECRYPT_MODE, secretKey, ivParameterSpec)
        return CipherInputStream(payload, cipher)
    }
}