package org.dronda.lib.jvm.io

import java.io.IOException
import java.io.InputStream


/**
 * A `ListSequenceInputStream` represents
 * the logical concatenation of other input
 * streams. It starts out with an ordered
 * collection of input streams and reads from
 * the first one until end of file is reached,
 * whereupon it reads from the second one,
 * and so on, until end of file is reached
 * on the last of the contained input streams.
 *
 * @author  Author van Hoff, Todd Bednarczyk
 * @since   1.0
 */
public class CollectionSequenceInputStream(streams: List<InputStream>) : InputStream() {
    private val iterator = streams.iterator()
    private var inputStream: InputStream? = null

    init {
        peekNextStream()
    }

    /**
     * Initializes a newly
     * created `SequenceInputStream`
     * by remembering the two arguments, which
     * will be read in order, first `s1`
     * and then `s2`, to provide the
     * bytes to be read from this `SequenceInputStream`.
     *
     * @param   s1   the first input stream to read.
     * @param   s2   the second input stream to read.
     */
    public constructor(s1: InputStream, s2: InputStream): this(
        ArrayList<InputStream>(2).apply {
            add(s1)
            add(s2)
        }
    )

    /**
     * Continues reading in the next stream if an EOF is reached.
     */
    @Throws(IOException::class)
    public fun nextStream() {
        inputStream?.close()
        peekNextStream()
    }

    private fun peekNextStream() {
        inputStream = if (iterator.hasNext()) {
            iterator.next()
        } else {
            null
        }
    }

    /**
     * Returns an estimate of the number of bytes that can be read (or
     * skipped over) from the current underlying input stream without
     * blocking by the next invocation of a method for the current
     * underlying input stream. The next invocation might be
     * the same thread or another thread.  A single read or skip of this
     * many bytes will not block, but may read or skip fewer bytes.
     *
     *
     * This method simply calls `available` of the current underlying
     * input stream and returns the result.
     *
     * @return an estimate of the number of bytes that can be read (or
     * skipped over) from the current underlying input stream
     * without blocking or `0` if this input stream
     * has been closed by invoking its [.close] method
     * @exception  IOException  if an I/O error occurs.
     *
     * @since   1.1
     */
    @Throws(IOException::class)
    override fun available(): Int {
        return if (inputStream == null) {
            0 // no way to signal EOF from available()
        } else inputStream!!.available()
    }

    /**
     * Reads the next byte of data from this input stream. The byte is
     * returned as an `int` in the range `0` to
     * `255`. If no byte is available because the end of the
     * stream has been reached, the value `-1` is returned.
     * This method blocks until input data is available, the end of the
     * stream is detected, or an exception is thrown.
     *
     *
     * This method
     * tries to read one character from the current substream. If it
     * reaches the end of the stream, it calls the `close`
     * method of the current substream and begins reading from the next
     * substream.
     *
     * @return     the next byte of data, or `-1` if the end of the
     * stream is reached.
     * @exception  IOException  if an I/O error occurs.
     */
    @Throws(IOException::class)
    override fun read(): Int {
        while (inputStream != null) {
            val c = inputStream!!.read()
            if (c != -1) {
                return c
            }
            nextStream()
        }
        return -1
    }

    /**
     * Reads up to `len` bytes of data from this input stream
     * into an array of bytes.  If `len` is not zero, the method
     * blocks until at least 1 byte of input is available; otherwise, no
     * bytes are read and `0` is returned.
     *
     *
     * The `read` method of `SequenceInputStream`
     * tries to read the data from the current substream. If it fails to
     * read any characters because the substream has reached the end of
     * the stream, it calls the `close` method of the current
     * substream and begins reading from the next substream.
     *
     * @param      b     the buffer into which the data is read.
     * @param      off   the start offset in array `b`
     * at which the data is written.
     * @param      len   the maximum number of bytes read.
     * @return     int   the number of bytes read.
     * @exception  NullPointerException If `b` is `null`.
     * @exception  IndexOutOfBoundsException If `off` is negative,
     * `len` is negative, or `len` is greater than
     * `b.length - off`
     * @exception  IOException  if an I/O error occurs.
     */
    @Throws(IOException::class)
    override fun read(b: ByteArray, off: Int, len: Int): Int {
        if (inputStream == null) {
            return -1
        } else if (off < 0 || len < 0 || len > b.size - off) {
            throw IndexOutOfBoundsException()
        } else if (len == 0) {
            return 0
        }
        do {
            val n = inputStream!!.read(b, off, len)
            if (n > 0) {
                return n
            }
            nextStream()
        } while (inputStream != null)
        return -1
    }

    /**
     * Closes this input stream and releases any system resources
     * associated with the stream.
     * A closed `SequenceInputStream`
     * cannot  perform input operations and cannot
     * be reopened.
     *
     *
     * If this stream was created
     * from an enumeration, all remaining elements
     * are requested from the enumeration and closed
     * before the `close` method returns.
     *
     * @exception  IOException  if an I/O error occurs.
     */
    @Throws(IOException::class)
    override fun close() {
        do {
            nextStream()
        } while (inputStream != null)
    }
}
