package org.dronda.lib.ktor.auth

import com.auth0.jwt.algorithms.Algorithm
import io.ktor.server.application.Application
import io.ktor.server.application.install
import io.ktor.server.auth.Authentication
import io.ktor.server.auth.authenticate
import io.ktor.server.auth.jwt.JWTPrincipal
import io.ktor.server.auth.jwt.jwt
import io.ktor.server.routing.Route

public const val JWT_AUTH: String = "auth-jwt"

public fun Route.authenticateWithJwt(
    build: Route.() -> Unit
) {
    authenticate(JWT_AUTH, build = build)
}

public fun Application.installAuth(jwtCredentialsProvider: JWTCredentials) {
    install(Authentication) {
        jwt(JWT_AUTH) {
            realm = jwtCredentialsProvider.realm
            verifier(
                jwtCredentialsProvider.issuer,
                jwtCredentialsProvider.audience,
                Algorithm.RSA256(
                    jwtCredentialsProvider.publicKey,
                    jwtCredentialsProvider.privateKey
                )
            ) {
                acceptLeeway(3)
            }

            validate { jwtCredential ->
                val conditions = listOf(
                    jwtCredential.issuer == jwtCredentialsProvider.issuer,
                    jwtCredential.audience.contains(jwtCredentialsProvider.audience),
                ).all { it }

                if (conditions) {
                    JWTPrincipal(jwtCredential.payload)
                } else {
                    null
                }
            }
        }
    }
}