package org.dronda.lib.ktor.auth

import com.auth0.jwk.JwkProviderBuilder
import com.auth0.jwt.algorithms.Algorithm
import io.ktor.server.application.Application
import io.ktor.server.application.install
import io.ktor.server.auth.Authentication
import io.ktor.server.auth.AuthenticationConfig
import io.ktor.server.auth.authenticate
import io.ktor.server.auth.jwt.JWTPrincipal
import io.ktor.server.auth.jwt.jwt
import io.ktor.server.routing.Routing
import java.util.concurrent.TimeUnit
import kotlin.time.Duration.Companion.hours
import kotlin.time.toJavaDuration

public const val JWT_AUTH: String = "auth-jwt"

public fun Routing.authenticateWithJwt(
    build: Routing.() -> Unit
) {
    authenticate(JWT_AUTH, build = build)
}

public fun Application.installAuth(jwtCredentialsProvider: JWTCredentials) {
    install(Authentication) {
        jwt(JWT_AUTH) {
            realm = jwtCredentialsProvider.realm
            verifier(
                jwtCredentialsProvider.issuer,
                jwtCredentialsProvider.audience,
                Algorithm.RSA256(
                    jwtCredentialsProvider.publicKey,
                    jwtCredentialsProvider.privateKey
                )
            ) {
                acceptLeeway(3)
            }

            validate { jwtCredential ->
                val conditions = listOf(
                    jwtCredential.issuer == jwtCredentialsProvider.issuer,
                    jwtCredential.audience.contains(jwtCredentialsProvider.audience),
                ).all { it }

                if (conditions) {
                    JWTPrincipal(jwtCredential.payload)
                } else {
                    null
                }
            }
        }
    }
}

public fun AuthenticationConfig.installJwtAuth(issuer: String, realm: String, audience: String) {
    val jwkProvider = JwkProviderBuilder(issuer)
        .cached(10, 12.hours.toJavaDuration())
        .rateLimited(10, 1, TimeUnit.MINUTES)
        .build()

    jwt(JWT_AUTH) {
        this.realm = realm
        verifier(
            jwkProvider,
            issuer,
        ) {
            acceptLeeway(3)
        }

        validate { jwtCredential ->
            val conditions = listOf(
                jwtCredential.issuer == issuer,
                jwtCredential.audience.contains(audience),
            ).all { it }

            if (conditions) {
                JWTPrincipal(jwtCredential.payload)
            } else {
                null
            }
        }
    }
}
