package org.drools.compiler.integrationtests;

import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.drools.compiler.Cheese;
import org.drools.compiler.CommonTestMethodBase;
import org.drools.compiler.Person;
import org.drools.core.FactHandle;
import org.drools.core.base.ClassObjectType;
import org.drools.core.base.DroolsQuery;
import org.drools.core.common.*;
import org.drools.core.reteoo.*;
import org.drools.core.util.FastIterator;
import org.drools.core.util.index.LeftTupleIndexHashTable;
import org.drools.core.util.index.LeftTupleList;
import org.drools.core.util.index.RightTupleIndexHashTable;
import org.drools.core.util.index.RightTupleList;
import org.drools.core.impl.KnowledgeBaseImpl;
import org.drools.core.impl.StatefulKnowledgeSessionImpl;
import org.drools.core.rule.IndexableConstraint;
import org.junit.Test;
import org.kie.api.runtime.rule.Row;
import org.kie.api.runtime.rule.Variable;
import org.kie.api.runtime.rule.ViewChangedEventListener;
import org.kie.internal.KnowledgeBase;
import org.kie.internal.KnowledgeBaseFactory;
import org.kie.internal.builder.KnowledgeBuilder;
import org.kie.internal.builder.KnowledgeBuilderErrors;
import org.kie.internal.builder.KnowledgeBuilderFactory;
import org.kie.api.definition.type.FactType;
import org.kie.internal.io.ResourceFactory;
import org.kie.internal.runtime.StatefulKnowledgeSession;
import org.kie.api.io.ResourceType;


public class IndexingTest extends CommonTestMethodBase {

    @Test(timeout=10000)
    public void testBuildsIndexedAlphaNodes() {
        String drl = "";
        drl += "package org.drools.compiler.test\n";
        drl += "import org.drools.compiler.Person\n";
        drl += "rule test1\n";
        drl += "when\n";
        drl += "   Person(name == \"Mark\", age == 37)\n";
        drl += "   Person(name == \"Mark\", happy == true)\n";
        drl += "then\n";
        drl += "end\n";

        KnowledgeBase kbase = loadKnowledgeBaseFromString( drl );

        ObjectTypeNode otn = getObjectTypeNode(kbase, Person.class );
        ReteooWorkingMemoryInterface wm = ((StatefulKnowledgeSessionImpl)kbase.newStatefulKnowledgeSession()).session;

        AlphaNode alphaNode1 = ( AlphaNode ) otn.getSinkPropagator().getSinks()[0];
        CompositeObjectSinkAdapter sinkAdapter = (CompositeObjectSinkAdapter)alphaNode1.getSinkPropagator();
        ObjectSinkNodeList hashableSinks = sinkAdapter.getHashableSinks();
        assertNotNull(hashableSinks);
        assertEquals(2, hashableSinks.size());

        AlphaNode alphaNode2 = ( AlphaNode ) alphaNode1.getSinkPropagator().getSinks()[0];
        assertSame(hashableSinks.getFirst(), alphaNode2);

        AlphaNode alphaNode3 = ( AlphaNode ) alphaNode1.getSinkPropagator().getSinks()[1];
        assertSame(hashableSinks.getLast(), alphaNode3);
    }

    @Test(timeout=10000)
    public void testBuildsIndexedMemory() {
        // tests indexes are correctly built        
        String drl = "";
        drl += "package org.drools.compiler.test\n";
        drl += "import org.drools.compiler.Person\n";
        drl += "global java.util.List list\n";
        drl += "rule test1\n";
        drl += "when\n";
        drl += "   $p1  : Person($name : name )\n";
        drl += "   $p2 : Person(name == $name)\n"; //indexed
        drl += "   $p3 : Person(name == $p1.name)\n"; //indexed
        drl += "   $p4 : Person(address.street == $p1.address.street)\n"; //not indexed
        drl += "   $p5 : Person(address.street == $p1.name)\n";  // indexed
        //drl += "   $p6 : Person( $name == name)\n"; // not indexed and won't compile
        drl += "   $p7 : Person(addresses[\"key\"].street == $p1.name)\n";  // indexed
        drl += "   $p8 : Person(addresses[0].street == $p1.name)\n";  // indexed
        drl += "   $p9 : Person(name == $p1.address.street)\n"; //not indexed
        drl += "   $p10 : Person(addresses[0].street + 'xx' == $p1.name)\n";  // indexed
        drl += "   $p11 : Person(addresses[$p1].street == $p1.name)\n";  // not indexed
        drl += "then\n";
        drl += "end\n";

        KnowledgeBase kbase = loadKnowledgeBaseFromString( drl );
        
        ObjectTypeNode node = getObjectTypeNode(kbase, Person.class );
        ReteooWorkingMemoryInterface wm = ((StatefulKnowledgeSessionImpl)kbase.newStatefulKnowledgeSession()).session;
        
        LeftInputAdapterNode liaNode = (LeftInputAdapterNode) node.getSinkPropagator().getSinks()[0];
        JoinNode j2 = ( JoinNode ) liaNode.getSinkPropagator().getSinks()[0]; // $p2
        JoinNode j3 = ( JoinNode ) j2.getSinkPropagator().getSinks()[0];  // $p3
        JoinNode j4 = ( JoinNode ) j3.getSinkPropagator().getSinks()[0];  // $p4
        JoinNode j5 = ( JoinNode ) j4.getSinkPropagator().getSinks()[0];  // $p5
        //JoinNode j6 = ( JoinNode ) j5.getSinkPropagator().getSinks()[0];  // $p6 // won't compile
        JoinNode j7 = ( JoinNode ) j5.getSinkPropagator().getSinks()[0];  // $p7
        JoinNode j8 = ( JoinNode ) j7.getSinkPropagator().getSinks()[0];  // $p8
        JoinNode j9 = ( JoinNode ) j8.getSinkPropagator().getSinks()[0];  // $p9
        JoinNode j10 = ( JoinNode ) j9.getSinkPropagator().getSinks()[0];  // $p10
        JoinNode j11 = ( JoinNode ) j10.getSinkPropagator().getSinks()[0];  // $p11
        
        SingleBetaConstraints c = ( SingleBetaConstraints ) j2.getRawConstraints();
        assertEquals( "$name", ((IndexableConstraint)c.getConstraint()).getFieldIndex().getDeclaration().getIdentifier() );
        assertTrue( c.isIndexed() );        
        BetaMemory bm = ( BetaMemory ) wm.getNodeMemory( j2 );
        assertTrue( bm.getLeftTupleMemory() instanceof LeftTupleIndexHashTable);
        assertTrue( bm.getRightTupleMemory() instanceof RightTupleIndexHashTable );
        
        c = ( SingleBetaConstraints ) j3.getRawConstraints();
        assertEquals( "name", ((IndexableConstraint)c.getConstraint()).getFieldIndex().getDeclaration().getIdentifier() );
        assertTrue( c.isIndexed() );   
        bm = ( BetaMemory ) wm.getNodeMemory( j3 );
        assertTrue( bm.getLeftTupleMemory() instanceof LeftTupleIndexHashTable );
        assertTrue( bm.getRightTupleMemory() instanceof RightTupleIndexHashTable );
        
        c = ( SingleBetaConstraints ) j4.getRawConstraints();
        assertEquals("$p1", c.getConstraint().getRequiredDeclarations()[0].getIdentifier());
        assertFalse( c.isIndexed() );   
        bm = ( BetaMemory ) wm.getNodeMemory( j4 );
        assertTrue( bm.getLeftTupleMemory() instanceof LeftTupleList );
        assertTrue( bm.getRightTupleMemory() instanceof RightTupleList );
        
        c = ( SingleBetaConstraints ) j5.getRawConstraints();
        assertEquals("name", ((IndexableConstraint)c.getConstraint()).getFieldIndex().getDeclaration().getIdentifier());
        assertTrue( c.isIndexed() );   
        bm = ( BetaMemory ) wm.getNodeMemory( j5 );
        assertTrue( bm.getLeftTupleMemory() instanceof LeftTupleIndexHashTable );
        assertTrue( bm.getRightTupleMemory() instanceof RightTupleIndexHashTable );   
        
        // won't compile
//        c = ( SingleBetaConstraints ) j6.getRawConstraints();
//        assertEquals( "name", ((VariableConstraint)c.getConstraint()).getRequiredDeclarations()[0].getIdentifier() );
//        assertFalse( c.isIndexed() );   
//        bm = ( BetaMemory ) wm.getNodeMemory( j6 );
//        assertTrue( bm.getLeftTupleMemory() instanceof LeftTupleList );
//        assertTrue( bm.getRightTupleMemory() instanceof RightTupleList );   
        
        c = ( SingleBetaConstraints ) j7.getRawConstraints();
        assertEquals("name", ((IndexableConstraint)c.getConstraint()).getFieldIndex().getDeclaration().getIdentifier());
        assertTrue( c.isIndexed() );   
        bm = ( BetaMemory ) wm.getNodeMemory( j7 );
        assertTrue( bm.getLeftTupleMemory() instanceof LeftTupleIndexHashTable );
        assertTrue( bm.getRightTupleMemory() instanceof RightTupleIndexHashTable );   
        
        c = ( SingleBetaConstraints ) j8.getRawConstraints();
        assertEquals("name", ((IndexableConstraint)c.getConstraint()).getFieldIndex().getDeclaration().getIdentifier());
        assertTrue( c.isIndexed() );   
        bm = ( BetaMemory ) wm.getNodeMemory( j8 );
        assertTrue( bm.getLeftTupleMemory() instanceof LeftTupleIndexHashTable );
        assertTrue( bm.getRightTupleMemory() instanceof RightTupleIndexHashTable );     
        
        c = ( SingleBetaConstraints ) j9.getRawConstraints();
        assertEquals("$p1", c.getConstraint().getRequiredDeclarations()[0].getIdentifier());
        assertFalse( c.isIndexed() );   
        bm = ( BetaMemory ) wm.getNodeMemory( j9 );
        assertTrue( bm.getLeftTupleMemory() instanceof LeftTupleList );
        assertTrue( bm.getRightTupleMemory() instanceof RightTupleList );  
        
        c = ( SingleBetaConstraints ) j10.getRawConstraints();
        assertEquals("name", ((IndexableConstraint)c.getConstraint()).getFieldIndex().getDeclaration().getIdentifier());
        assertTrue( c.isIndexed() );   
        bm = ( BetaMemory ) wm.getNodeMemory( j10 );
        assertTrue( bm.getLeftTupleMemory() instanceof LeftTupleIndexHashTable );
        assertTrue( bm.getRightTupleMemory() instanceof RightTupleIndexHashTable );  
        
        c = ( SingleBetaConstraints ) j11.getRawConstraints();
        assertEquals("$p1", c.getConstraint().getRequiredDeclarations()[0].getIdentifier());
        assertFalse( c.isIndexed() );   
        bm = ( BetaMemory ) wm.getNodeMemory( j11 );
        assertTrue( bm.getLeftTupleMemory() instanceof LeftTupleList);
        assertTrue( bm.getRightTupleMemory() instanceof RightTupleList );          
    }
    
    @Test(timeout=10000)
    public void testIndexingOnQueryUnification() throws Exception {
        String str = "";
        str += "package org.drools.compiler.test  \n";
        str += "import org.drools.compiler.Person \n";
        str += "query peeps( String $name, String $likes, String $street) \n";
        str += "    $p : Person( $name := name, $likes := likes, $street := address.street ) \n";
        str += "end\n";

        KnowledgeBase kbase = loadKnowledgeBaseFromString( str );
        
        List<ObjectTypeNode> nodes = ((InternalRuleBase)((KnowledgeBaseImpl)kbase).ruleBase).getRete().getObjectTypeNodes();
        ObjectTypeNode node = null;
        for ( ObjectTypeNode n : nodes ) {
            if ( ((ClassObjectType)n.getObjectType()).getClassType() == DroolsQuery.class ) {
                node = n;
                break;
            }
        }    
        
        ReteooWorkingMemoryInterface wm = ((StatefulKnowledgeSessionImpl)kbase.newStatefulKnowledgeSession()).session;
        
        AlphaNode alphanode = ( AlphaNode ) node.getSinkPropagator().getSinks()[0];
        LeftInputAdapterNode liaNode = (LeftInputAdapterNode) alphanode.getSinkPropagator().getSinks()[0];
        JoinNode j = ( JoinNode ) liaNode.getSinkPropagator().getSinks()[0]; // $p2
        
        TripleNonIndexSkipBetaConstraints c = ( TripleNonIndexSkipBetaConstraints ) j.getRawConstraints();
        //assertEquals( "$name", ((VariableConstraint)c.getConstraint()).getRequiredDeclarations()[0].getIdentifier() );
        assertTrue( c.isIndexed() );        
        BetaMemory bm = ( BetaMemory ) wm.getNodeMemory( j );
        assertTrue( bm.getLeftTupleMemory() instanceof LeftTupleIndexHashTable );
        assertTrue( bm.getRightTupleMemory() instanceof RightTupleIndexHashTable );        
    }

    @Test(timeout=10000)
    public void testIndexingOnQueryUnificationWithNot() throws Exception {
        String str = "";
        str += "package org.drools.compiler.test  \n";
        str += "import org.drools.compiler.Person \n";
        str += "query peeps( String $name, int $age ) \n";
        str += "    not $p2 : Person( $name := name, age > $age ) \n";
        str += "end\n";

        KnowledgeBase kbase = loadKnowledgeBaseFromString( str );

        List<ObjectTypeNode> nodes = ((InternalRuleBase)((KnowledgeBaseImpl)kbase).ruleBase).getRete().getObjectTypeNodes();
        ObjectTypeNode node = null;
        for ( ObjectTypeNode n : nodes ) {
            if ( ((ClassObjectType)n.getObjectType()).getClassType() == DroolsQuery.class ) {
                node = n;
                break;
            }
        }

        ReteooWorkingMemoryInterface wm = ((StatefulKnowledgeSessionImpl)kbase.newStatefulKnowledgeSession()).session;

        AlphaNode alphanode = ( AlphaNode ) node.getSinkPropagator().getSinks()[0];
        LeftInputAdapterNode liaNode = (LeftInputAdapterNode) alphanode.getSinkPropagator().getSinks()[0];

        NotNode n = (NotNode) liaNode.getSinkPropagator().getSinks()[0];

        DoubleNonIndexSkipBetaConstraints c = (DoubleNonIndexSkipBetaConstraints) n.getRawConstraints();
        //assertEquals( "$name", ((VariableConstraint)c.getConstraint()).getRequiredDeclarations()[0].getIdentifier() );
        assertTrue( c.isIndexed() );
        BetaMemory bm = ( BetaMemory ) wm.getNodeMemory( n );
        System.out.println( bm.getLeftTupleMemory().getClass() );
        System.out.println( bm.getRightTupleMemory().getClass() );
        assertTrue(bm.getLeftTupleMemory() instanceof LeftTupleIndexHashTable);
        assertTrue( bm.getRightTupleMemory() instanceof RightTupleIndexHashTable );


        final Map<String, Integer> map = new HashMap<String, Integer>();
        map.put("inserted", new Integer(0));
        map.put("deleted", new Integer(0));
        map.put("updated", new Integer(0));
        wm.openLiveQuery("peeps", new Object[] {Variable.v, 99 }, new ViewChangedEventListener() {
            @Override
            public void rowInserted(Row row) {
                System.out.println( "inserted" );
                Integer integer = map.get("inserted");
                map.put("inserted", integer.intValue() + 1 );
            }

            @Override
            public void rowDeleted(Row row) {
                System.out.println( "deleted" );
                Integer integer = map.get("deleted");
                map.put("deleted", integer.intValue() + 1 );
            }

            @Override
            public void rowUpdated(Row row) {
                System.out.println( "updated" );
                Integer integer = map.get("updated");
                map.put("updated", integer.intValue() + 1 );
            }
        });

        System.out.println( "inserted: " + map.get("inserted"));
        System.out.println( "deleted: " + map.get("deleted"));
        System.out.println( "updated: " + map.get("updated"));

        Map<String, InternalFactHandle> peeps = new HashMap<String, InternalFactHandle>();

        Person p = null;
        InternalFactHandle fh = null;

        int max = 3;

        // 1 matched, prior to any insertions
        assertEquals( 1, map.get("inserted").intValue() );
        assertEquals( 0, map.get("deleted").intValue() );
        assertEquals( 0, map.get("updated").intValue() );

        // x0 is the blocker
        for ( int i = 0; i < max; i++ ) {
            p = new Person( "x" + i, 100);
            fh = ( InternalFactHandle ) wm.insert( p );
            wm.fireAllRules();
            peeps.put(p.getName(), fh);
        }

        // insertions case 1 deletion
        assertEquals( 1, map.get("inserted").intValue() );
        assertEquals( 1, map.get("deleted").intValue() );
        assertEquals( 0, map.get("updated").intValue() );

        // each x is blocker in turn up to x99
        for ( int i = 0; i < (max-1); i++ ) {
            fh = peeps.get("x" + i);
            p = (Person) fh.getObject();
            p.setAge( 90 );
            wm.update( fh, p );
            wm.fireAllRules();
            assertEquals( "i=" + i, 1, map.get("inserted").intValue() ); // make sure this doesn't change
        }

        // no change
        assertEquals( 1, map.get("inserted").intValue() );
        assertEquals( 1, map.get("deleted").intValue() );
        assertEquals( 0, map.get("updated").intValue() );

        // x99 is still the blocker, everything else is just added
        for ( int i = 0; i < (max-1); i++ ) {
            fh = peeps.get("x" + i);
            p = (Person) fh.getObject();
            p.setAge( 102 );
            wm.update( fh, p );
            wm.fireAllRules();
            assertEquals( "i=" + i, 1, map.get("inserted").intValue() ); // make sure this doesn't change
        }

        // no change
        assertEquals( 1, map.get("inserted").intValue() );
        assertEquals( 1, map.get("deleted").intValue() );
        assertEquals( 0, map.get("updated").intValue() );

        // x99 is still the blocker
        for ( int i = (max-2); i >= 0; i-- ) {
            fh = peeps.get("x" + i);
            p = (Person) fh.getObject();
            p.setAge( 90 );
            wm.update( fh, p );
            wm.fireAllRules();
            assertEquals( "i=" + i, 1, map.get("inserted").intValue() ); // make sure this doesn't change
        }

        // move x99, should no longer be a blocker, now it can increase
        fh = peeps.get("x" + (max-1));
        p = (Person) fh.getObject();
        p.setAge( 90 );
        wm.update( fh, p );
        wm.fireAllRules();
        assertEquals( 2, map.get("inserted").intValue() );
    }

    @Test(timeout=10000)
    public void testFullFastIteratorResume() throws Exception {
        String str = "";
        str += "package org.drools.compiler.test  \n";
        str += "import org.drools.compiler.Person \n";
        str += "query peeps( String $name, int $age ) \n";
        str += "    not $p2 : Person( $name := name, age > $age ) \n";
        str += "end\n";

        KnowledgeBase kbase = loadKnowledgeBaseFromString( str );

        List<ObjectTypeNode> nodes = ((InternalRuleBase)((KnowledgeBaseImpl)kbase).ruleBase).getRete().getObjectTypeNodes();
        ObjectTypeNode node = null;
        for ( ObjectTypeNode n : nodes ) {
            if ( ((ClassObjectType)n.getObjectType()).getClassType() == DroolsQuery.class ) {
                node = n;
                break;
            }
        }

        ReteooWorkingMemoryInterface wm = ((StatefulKnowledgeSessionImpl)kbase.newStatefulKnowledgeSession()).session;

        AlphaNode alphanode = ( AlphaNode ) node.getSinkPropagator().getSinks()[0];
        LeftInputAdapterNode liaNode = (LeftInputAdapterNode) alphanode.getSinkPropagator().getSinks()[0];

        NotNode n = (NotNode) liaNode.getSinkPropagator().getSinks()[0];

        DoubleNonIndexSkipBetaConstraints c = (DoubleNonIndexSkipBetaConstraints) n.getRawConstraints();
        //assertEquals( "$name", ((VariableConstraint)c.getConstraint()).getRequiredDeclarations()[0].getIdentifier() );
        assertTrue( c.isIndexed() );
        BetaMemory bm = ( BetaMemory ) wm.getNodeMemory( n );
        System.out.println( bm.getLeftTupleMemory().getClass() );
        System.out.println( bm.getRightTupleMemory().getClass() );
        assertTrue(bm.getLeftTupleMemory() instanceof LeftTupleIndexHashTable);
        assertTrue( bm.getRightTupleMemory() instanceof RightTupleIndexHashTable );


        final Map<String, Integer> map = new HashMap<String, Integer>();
        map.put("inserted", new Integer(0));
        map.put("deleted", new Integer(0));
        map.put("updated", new Integer(0));
        wm.openLiveQuery("peeps", new Object[] {Variable.v, 99 }, new ViewChangedEventListener() {
            @Override
            public void rowInserted(Row row) {
            }

            @Override
            public void rowDeleted(Row row) {
            }

            @Override
            public void rowUpdated(Row row) {
            }
        });

        Map<String, InternalFactHandle> peeps = new HashMap<String, InternalFactHandle>();

        Person p = new Person( "x0", 100);
        InternalFactHandle fh = ( InternalFactHandle ) wm.insert( p );

        peeps.put(p.getName(), fh);

        for ( int i = 1; i < 100; i++ ) {
            p = new Person( "x" + i, 101);
            fh = ( InternalFactHandle ) wm.insert( p );
            wm.fireAllRules();
            peeps.put(p.getName(), fh);
        }

        List<RightTuple> list = new ArrayList<RightTuple>(100);
        FastIterator it = n.getRightIterator( bm.getRightTupleMemory() );
        for ( RightTuple rt =n.getFirstRightTuple(null, bm.getRightTupleMemory(), null, it); rt != null; rt = (RightTuple)it.next(rt) ) {
            list.add(rt);
        }
        assertEquals( 100, list.size() );

        // check we can resume from each entry in the list above.
        for ( int i = 0; i < 100; i++ ) {
            RightTuple rightTuple = list.get(i);
            it = n.getRightIterator( bm.getRightTupleMemory(), rightTuple ); // resumes from the current rightTuple
            int j = i + 1;
            for ( RightTuple rt = ( RightTuple ) it.next(rightTuple); rt != null; rt = (RightTuple)it.next(rt) ) {
                assertSame( list.get(j), rt);
                j++;
            }
        }
    }

    public ObjectTypeNode getObjectTypeNode(KnowledgeBase kbase, Class<?> nodeClass) {
        List<ObjectTypeNode> nodes = ((InternalRuleBase)((KnowledgeBaseImpl)kbase).ruleBase).getRete().getObjectTypeNodes();
        for ( ObjectTypeNode n : nodes ) {
            if ( ((ClassObjectType)n.getObjectType()).getClassType() == nodeClass ) {
                return n;
            }
        }
        return null;
    }
    @Test(timeout=10000)
    public void testRangeIndex() {
        String str = "import org.drools.compiler.*;\n" +
                "rule R1\n" +
                "when\n" +
                "   $s : String()" +
                "   exists Cheese( type > $s )\n" +
                "then\n" +
                "   System.out.println( $s );\n" +
                "end\n";

        KnowledgeBase kbase = loadKnowledgeBaseFromString(str);
        StatefulKnowledgeSession ksession = kbase.newStatefulKnowledgeSession();

        ksession.insert( "cheddar" );
        ksession.insert( "gorgonzola" );
        ksession.insert( "stilton" );
        ksession.insert( new Cheese( "gorgonzola", 10 ) );
        assertEquals(1, ksession.fireAllRules());
        ksession.dispose();
    }

    @Test(timeout=10000)
    public void testRangeIndex2() {
        String str = "import org.drools.compiler.*;\n" +
                "rule R1\n" +
                "when\n" +
                "   $s : String()" +
                "   exists Cheese( type < $s )\n" +
                "then\n" +
                "   System.out.println( $s );\n" +
                "end\n";

        KnowledgeBase kbase = loadKnowledgeBaseFromString(str);
        StatefulKnowledgeSession ksession = kbase.newStatefulKnowledgeSession();

        ksession.insert( "gorgonzola" );
        ksession.insert( new Cheese( "cheddar", 10 ) );
        ksession.insert( new Cheese( "gorgonzola", 10 ) );
        ksession.insert( new Cheese( "stilton", 10 ) );
        assertEquals(1, ksession.fireAllRules());
        ksession.dispose();
    }

    @Test(timeout=10000)
    public void testNotNode() {
        String str = "import org.drools.compiler.*;\n" +
                "rule R1 salience 10\n" +
                "when\n" +
                "   Person( $age : age )" +
                "   not Cheese( price < $age )\n" +
                "then\n" +
                "   System.out.println( $age );\n" +
                "end\n" +
                "rule R2 salience 1\n" +
                "when\n" +
                "   $p : Person( age == 10 )" +
                "then\n" +
                "   modify($p) { setAge(15); }\n" +
                "end\n";

        KnowledgeBase kbase = loadKnowledgeBaseFromString(str);
        StatefulKnowledgeSession ksession = kbase.newStatefulKnowledgeSession();

        ksession.insert( new Person( "mario", 10 ) );
        ksession.insert( new Cheese( "gorgonzola", 20 ) );
        assertEquals(3, ksession.fireAllRules());
        ksession.dispose();
    }

    @Test(timeout=10000)
    public void testNotNodeModifyRight() {
        String str = "import org.drools.compiler.*;\n" +
                "rule R1 salience 10 when\n" +
                "   Person( $age : age )\n" +
                "   not Cheese( price < $age )\n" +
                "then\n" +
                "   System.out.println( $age );\n" +
                "end\n" +
                "rule R3 salience 5 when\n" +
                "   $c : Cheese( price == 8 )\n" +
                "then\n" +
                "   modify($c) { setPrice(15); }\n" +
                "end\n";

        KnowledgeBase kbase = loadKnowledgeBaseFromString(str);
        StatefulKnowledgeSession ksession = kbase.newStatefulKnowledgeSession();

        ksession.insert( new Person( "A", 10 ) );
        ksession.insert( new Cheese( "C1", 20 ) );
        ksession.insert( new Cheese( "C2", 8 ) );
        assertEquals(2, ksession.fireAllRules());
        ksession.dispose();
    }

    @Test(timeout=10000)
    public void testRange() {
        String str = "import org.drools.compiler.*;\n" +
                "rule R1 salience 10 when\n" +
                "   Person( $age : age, $doubleAge : doubleAge )\n" +
                "   not Cheese( this.price > $age && < $doubleAge )\n" +
                "then\n" +
                "   System.out.println( $age );\n" +
                "end\n" +
                "rule R3 salience 5 when\n" +
                "   $c : Cheese( price == 15 )\n" +
                "then\n" +
                "   System.out.println( \"modify\" );\n" +
                "   modify($c) { setPrice(8); }\n" +
                "end\n";

        KnowledgeBase kbase = loadKnowledgeBaseFromString(str);
        StatefulKnowledgeSession ksession = kbase.newStatefulKnowledgeSession();

        ksession.insert( new Person( "A", 10 ) );
        ksession.insert( new Cheese( "C1", 30 ) );
        ksession.insert( new Cheese( "C2", 15 ) );
        assertEquals(2, ksession.fireAllRules());
        ksession.dispose();
    }

    @Test(timeout=10000)
    public void testRange2() throws Exception {
        String rule = "package org.drools.compiler.test\n" +
                "declare A\n" +
                "    a: int\n" +
                "end\n" +
                "declare B\n" +
                "    b: int\n" +
                "end\n" +
                "declare C\n" +
                "    c: int\n" +
                "end\n" +
                "rule R1 when\n" +
                "   A( $a : a )\n" +
                "   B( $b : b )\n" +
                "   exists C( c > $a && < $b )\n" +
                "then\n" +
                "   System.out.println( $a + \", \" + $b );\n" +
                "end";

        KnowledgeBase kbase = loadKnowledgeBaseFromString( rule );
        StatefulKnowledgeSession ksession = kbase.newStatefulKnowledgeSession();

        FactType aType = kbase.getFactType( "org.drools.compiler.test", "A" );
        FactType bType = kbase.getFactType( "org.drools.compiler.test", "B" );
        FactType cType = kbase.getFactType( "org.drools.compiler.test", "C" );

        Object a1 = aType.newInstance();
        aType.set( a1, "a", 5 );
        ksession.insert( a1 );
        Object a2 = aType.newInstance();
        aType.set( a2, "a", 11 );
        ksession.insert( a2 );

        Object b1 = bType.newInstance();
        bType.set( b1, "b", 10 );
        ksession.insert( b1 );
        Object b2 = bType.newInstance();
        bType.set( b2, "b", 6 );
        ksession.insert( b2 );

        Object c = cType.newInstance();
        cType.set( c, "c", 7 );
        ksession.insert( c );

        ksession.fireAllRules();
    }
}
