package org.kie.builder.impl;

import org.drools.builder.impl.KnowledgeBuilderImpl;
import org.drools.compiler.PackageBuilder;
import org.drools.io.impl.BaseResource;
import org.kie.KieServices;
import org.kie.builder.CompositeKnowledgeBuilder;
import org.kie.builder.IncrementalResults;
import org.kie.builder.model.KieBaseModel;
import org.kie.builder.KieBuilderSet;
import org.kie.builder.KnowledgeBuilder;
import org.kie.builder.KnowledgeBuilderError;
import org.kie.io.Resource;
import org.kie.io.ResourceConfiguration;
import org.kie.io.ResourceType;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.net.URL;
import java.util.Collection;

import static org.kie.builder.impl.AbstractKieModule.getResourceConfiguration;
import static org.kie.builder.impl.KieBuilderImpl.filterFileInKBase;

public class KieBuilderSetImpl implements KieBuilderSet {

    private final KieBuilderImpl kieBuilder;
    private final String[] files;

    public KieBuilderSetImpl(KieBuilderImpl kieBuilder, String[] files) {
        this.kieBuilder = kieBuilder;
        this.files = files;
    }

    @Override
    public IncrementalResults build() {
        for (String file : files) {
            kieBuilder.copySourceToTarget(file);
        }
        return buildChanges();
    }

    private IncrementalResults buildChanges() {
        IncrementalResultsImpl results = new IncrementalResultsImpl();

        InternalKieModule kieModule = (InternalKieModule) kieBuilder.getKieModuleIgnoringErrors();
        for (KieBaseModel kBaseModel : kieModule.getKieModuleModel().getKieBaseModels().values()) {
            KnowledgeBuilder kBuilder = kieModule.getKnowledgeBuilderForKieBase( kBaseModel.getName() );
            for ( KnowledgeBuilderError error : kBuilder.getErrors() ) {
                results.removeMessage( error );
            }

            CompositeKnowledgeBuilder ckbuilder = kBuilder.batch();

            PackageBuilder pkgBuilder = ((KnowledgeBuilderImpl)kBuilder).getPackageBuilder();
            boolean modified = false;
            for (String file : files) {
                String resourceName = file.startsWith(KieBuilderImpl.RESOURCES_ROOT) ?
                        file.substring(KieBuilderImpl.RESOURCES_ROOT.length()) :
                        file;

                // remove the objects generated by the old Resource
                modified = pkgBuilder.removeObjectsGeneratedFromResource(new DummyResource(resourceName)) || modified;

                // add the modified Resource
                modified = addResource(ckbuilder, kBaseModel, kieModule, resourceName) || modified;
            }

            if (modified) {
                ckbuilder.build();
                if ( kBuilder.hasErrors() ) {
                    for ( KnowledgeBuilderError error : kBuilder.getErrors() ) {
                        results.addMessage( error );
                    }
                    kBuilder.undo();
                } else {
                    KieServices.Factory.get().getRepository().addKieModule( kieModule );
                }
            }
        }
        return results;
    }

    private boolean addResource( CompositeKnowledgeBuilder ckbuilder,
                                 KieBaseModel kieBaseModel,
                                 InternalKieModule kieModule,
                                 String resourceName ) {
        byte[] bytes = kieModule.getBytes(resourceName);
        if (bytes == null) {
            return false;
        }
        Resource resource = KieServices.Factory.get().getResources().newByteArrayResource(bytes).setSourcePath(resourceName);

        if ( filterFileInKBase(kieBaseModel, resourceName) ) {
            ResourceConfiguration conf = getResourceConfiguration(kieModule, resourceName);
            if ( conf == null ) {
                ckbuilder.add( resource, ResourceType.determineResourceType(resourceName) );
            } else {
                ckbuilder.add( resource, ResourceType.determineResourceType(resourceName), conf );
            }
            return true;
        }
        return false;
    }

    public static class DummyResource extends BaseResource {
        public DummyResource(String resourceName) {
            setSourcePath(resourceName);
        }

        @Override
        public URL getURL() throws IOException {
            throw new UnsupportedOperationException("org.kie.builder.impl.KieBuilderSetImpl.DummyResource.getURL -> TODO");
        }

        @Override
        public boolean hasURL() {
            throw new UnsupportedOperationException("org.kie.builder.impl.KieBuilderSetImpl.DummyResource.hasURL -> TODO");
        }

        @Override
        public boolean isDirectory() {
            throw new UnsupportedOperationException("org.kie.builder.impl.KieBuilderSetImpl.DummyResource.isDirectory -> TODO");
        }

        @Override
        public Collection<Resource> listResources() {
            throw new UnsupportedOperationException("org.kie.builder.impl.KieBuilderSetImpl.DummyResource.listResources -> TODO");
        }

        @Override
        public long getLastModified() {
            throw new UnsupportedOperationException("org.kie.builder.impl.KieBuilderSetImpl.DummyResource.getLastModified -> TODO");
        }

        @Override
        public long getLastRead() {
            throw new UnsupportedOperationException("org.kie.builder.impl.KieBuilderSetImpl.DummyResource.getLastRead -> TODO");
        }

        @Override
        public InputStream getInputStream() throws IOException {
            throw new UnsupportedOperationException("org.kie.builder.impl.KieBuilderSetImpl.DummyResource.getInputStream -> TODO");
        }

        @Override
        public Reader getReader() throws IOException {
            throw new UnsupportedOperationException("org.kie.builder.impl.KieBuilderSetImpl.DummyResource.getReader -> TODO");
        }
    }
}