/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.graph;

import com.google.common.annotations.Beta;
import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.graph.Graph;
import com.google.common.graph.GraphBuilder;
import com.google.common.graph.Hypergraph;
import com.google.common.graph.MutableGraph;
import com.google.common.graph.MutableNetwork;
import com.google.common.graph.Network;
import com.google.common.graph.NetworkBuilder;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import javax.annotation.Nullable;

@Beta
public final class Graphs {
    private static final String GRAPH_FORMAT = "%s, nodes: %s, edges: %s";
    private static final String DIRECTED_FORMAT = "<%s -> %s>";
    private static final String UNDIRECTED_FORMAT = "[%s, %s]";

    private Graphs() {
    }

    public static <N> N oppositeNode(Network<N, ?> graph, Object edge, Object node) {
        N node2;
        if (graph instanceof Hypergraph) {
            throw new UnsupportedOperationException();
        }
        Preconditions.checkNotNull(node, "node");
        Iterator<N> incidentNodesIterator = graph.incidentNodes(edge).iterator();
        N node1 = incidentNodesIterator.next();
        N n = node2 = incidentNodesIterator.hasNext() ? incidentNodesIterator.next() : node1;
        if (node.equals(node1)) {
            return node2;
        }
        Preconditions.checkArgument(node.equals(node2), "Edge %s is not incident to node %s", edge, node);
        return node1;
    }

    public static <N, E> Set<E> parallelEdges(Network<N, E> graph, Object edge) {
        if (graph instanceof Hypergraph) {
            throw new UnsupportedOperationException();
        }
        Set<N> incidentNodes = graph.incidentNodes(edge);
        if (!graph.allowsParallelEdges()) {
            return ImmutableSet.of();
        }
        Iterator<N> incidentNodesIterator = incidentNodes.iterator();
        N node1 = incidentNodesIterator.next();
        N node2 = incidentNodesIterator.hasNext() ? incidentNodesIterator.next() : node1;
        return Sets.difference(graph.edgesConnecting(node1, node2), ImmutableSet.of(edge));
    }

    @CanIgnoreReturnValue
    public static <N, E> boolean addEdge(MutableNetwork<N, E> graph, E edge, Iterable<N> nodes) {
        Preconditions.checkNotNull(graph, "graph");
        Preconditions.checkNotNull(edge, "edge");
        Preconditions.checkNotNull(nodes, "nodes");
        if (graph instanceof Hypergraph) {
            return ((Hypergraph)((Object)graph)).addEdge(edge, nodes);
        }
        Iterator<N> nodesIterator = nodes.iterator();
        Preconditions.checkArgument(nodesIterator.hasNext(), "'graph' is not a Hypergraph, and 'nodes' has < 1 elements: %s", nodes);
        N node1 = nodesIterator.next();
        N node2 = nodesIterator.hasNext() ? nodesIterator.next() : node1;
        Preconditions.checkArgument(!nodesIterator.hasNext(), "'graph' is not a Hypergraph, and 'nodes' has > 2 elements: %s", nodes);
        return graph.addEdge(edge, node1, node2);
    }

    public static <N> MutableGraph<N> copyOf(Graph<N> graph) {
        return Graphs.copyOf(graph, Predicates.alwaysTrue());
    }

    public static <N> MutableGraph<N> copyOf(Graph<N> graph, Predicate<? super N> nodePredicate) {
        Preconditions.checkNotNull(graph, "graph");
        Preconditions.checkNotNull(nodePredicate, "nodePredicate");
        MutableGraph copy = GraphBuilder.from(graph).expectedNodeCount(graph.nodes().size()).build();
        for (N node : graph.nodes()) {
            if (!nodePredicate.apply(node)) continue;
            copy.addNode(node);
            for (N successor : graph.successors(node)) {
                if (!nodePredicate.apply(successor)) continue;
                copy.addEdge(node, successor);
            }
        }
        return copy;
    }

    public static <N> void mergeNodesFrom(Graph<N> original, MutableGraph<N> copy) {
        Graphs.mergeNodesFrom(original, copy, Predicates.alwaysTrue());
    }

    public static <N, E> void mergeNodesFrom(Graph<N> original, MutableGraph<N> copy, Predicate<? super N> nodePredicate) {
        Preconditions.checkNotNull(original, "original");
        Preconditions.checkNotNull(copy, "copy");
        Preconditions.checkNotNull(nodePredicate, "nodePredicate");
        for (N node : Sets.filter(original.nodes(), nodePredicate)) {
            copy.addNode(node);
        }
    }

    public static <N, E> MutableNetwork<N, E> copyOf(Network<N, E> graph) {
        return Graphs.copyOf(graph, Predicates.alwaysTrue(), Predicates.alwaysTrue());
    }

    public static <N, E> MutableNetwork<N, E> copyOf(Network<N, E> graph, Predicate<? super N> nodePredicate, Predicate<? super E> edgePredicate) {
        Preconditions.checkNotNull(graph, "graph");
        Preconditions.checkNotNull(nodePredicate, "nodePredicate");
        Preconditions.checkNotNull(edgePredicate, "edgePredicate");
        MutableNetwork copy = NetworkBuilder.from(graph).expectedNodeCount(graph.nodes().size()).expectedEdgeCount(graph.edges().size()).build();
        Graphs.mergeNodesFrom(graph, copy, nodePredicate);
        for (E edge : graph.edges()) {
            if (!edgePredicate.apply(edge)) continue;
            Set<N> incidentNodes = graph.incidentNodes(edge);
            if (!copy.nodes().containsAll(incidentNodes)) continue;
            Graphs.addEdge(copy, edge, incidentNodes);
        }
        return copy;
    }

    public static <N> void mergeNodesFrom(Graph<N> original, MutableNetwork<N, ?> copy) {
        Graphs.mergeNodesFrom(original, copy, Predicates.alwaysTrue());
    }

    public static <N, E> void mergeNodesFrom(Graph<N> original, MutableNetwork<N, ?> copy, Predicate<? super N> nodePredicate) {
        Preconditions.checkNotNull(original, "original");
        Preconditions.checkNotNull(copy, "copy");
        Preconditions.checkNotNull(nodePredicate, "nodePredicate");
        for (N node : Sets.filter(original.nodes(), nodePredicate)) {
            copy.addNode(node);
        }
    }

    public static <N, E> void mergeEdgesFrom(Network<N, E> original, MutableNetwork<N, E> copy) {
        Graphs.mergeEdgesFrom(original, copy, Predicates.alwaysTrue());
    }

    public static <N, E> void mergeEdgesFrom(Network<N, E> original, MutableNetwork<N, E> copy, Predicate<? super E> edgePredicate) {
        Preconditions.checkNotNull(original, "original");
        Preconditions.checkNotNull(copy, "copy");
        Preconditions.checkNotNull(edgePredicate, "edgePredicate");
        for (E edge : Sets.filter(original.edges(), edgePredicate)) {
            Graphs.addEdge(copy, edge, original.incidentNodes(edge));
        }
    }

    public static boolean equal(@Nullable Graph<?> graph1, @Nullable Graph<?> graph2) {
        if (graph1 instanceof Network && graph2 instanceof Network) {
            return Graphs.equal((Network)graph1, (Network)graph2);
        }
        if (graph1 instanceof Network || graph2 instanceof Network) {
            return false;
        }
        if (graph1 == graph2) {
            return true;
        }
        if (graph1 == null || graph2 == null) {
            return false;
        }
        if (!graph1.nodes().equals(graph2.nodes())) {
            return false;
        }
        for (Object node : graph1.nodes()) {
            if (!graph1.successors(node).equals(graph2.successors(node))) {
                return false;
            }
            boolean bothUndirected = !graph1.isDirected() && !graph2.isDirected();
            if (bothUndirected || graph1.predecessors(node).equals(graph2.predecessors(node))) continue;
            return false;
        }
        return true;
    }

    public static boolean equal(@Nullable Network<?, ?> graph1, @Nullable Network<?, ?> graph2) {
        if (graph1 == graph2) {
            return true;
        }
        if (graph1 == null || graph2 == null) {
            return false;
        }
        if (graph1.edges().size() != graph2.edges().size()) {
            return false;
        }
        if (!graph1.nodes().equals(graph2.nodes())) {
            return false;
        }
        for (Object node : graph1.nodes()) {
            if (!graph1.inEdges(node).equals(graph2.inEdges(node))) {
                return false;
            }
            boolean bothUndirected = !graph1.isDirected() && !graph2.isDirected();
            if (bothUndirected || graph1.outEdges(node).equals(graph2.outEdges(node))) continue;
            return false;
        }
        return true;
    }

    public static int hashCode(Graph<?> graph) {
        return graph instanceof Network ? Graphs.hashCode((Network)graph) : Graphs.nodeToAdjacentNodes(graph).hashCode();
    }

    public static int hashCode(Network<?, ?> graph) {
        return Graphs.nodeToIncidentEdges(graph).hashCode();
    }

    public static String toString(Graph<?> graph) {
        if (graph instanceof Network) {
            return Graphs.toString((Network)graph);
        }
        return String.format(GRAPH_FORMAT, Graphs.getPropertiesString(graph), graph.nodes(), Graphs.adjacentNodesString(graph));
    }

    public static String toString(Network<?, ?> graph) {
        return String.format(GRAPH_FORMAT, Graphs.getPropertiesString(graph), graph.nodes(), Maps.asMap(graph.edges(), Graphs.edgeToIncidentNodesString(graph)));
    }

    public static <E> Predicate<E> selfLoopPredicate(final Network<?, E> graph) {
        Preconditions.checkNotNull(graph, "graph");
        return new Predicate<E>(){

            @Override
            public boolean apply(E edge) {
                return graph.incidentNodes(edge).size() == 1;
            }
        };
    }

    private static <N> String adjacentNodesString(Graph<N> graph) {
        Preconditions.checkNotNull(graph, "graph");
        ArrayList<String> adjacencies = new ArrayList<String>();
        for (N node : graph.nodes()) {
            for (N successor : graph.successors(node)) {
                adjacencies.add(String.format(graph.isDirected() ? DIRECTED_FORMAT : UNDIRECTED_FORMAT, node, successor));
            }
        }
        return String.format("{%s}", Joiner.on(", ").join(adjacencies));
    }

    private static <N, E> Map<N, Set<E>> nodeToIncidentEdges(final Network<N, E> graph) {
        Preconditions.checkNotNull(graph, "graph");
        return Maps.asMap(graph.nodes(), new Function<N, Set<E>>(){

            @Override
            public Set<E> apply(N node) {
                return graph.incidentEdges(node);
            }
        });
    }

    private static <N> Map<N, Set<N>> nodeToAdjacentNodes(final Graph<N> graph) {
        Preconditions.checkNotNull(graph, "graph");
        return Maps.asMap(graph.nodes(), new Function<N, Set<N>>(){

            @Override
            public Set<N> apply(N node) {
                return graph.adjacentNodes(node);
            }
        });
    }

    private static Function<Object, String> edgeToIncidentNodesString(final Network<?, ?> graph) {
        if (graph.isDirected()) {
            return new Function<Object, String>(){

                @Override
                public String apply(Object edge) {
                    return String.format(Graphs.DIRECTED_FORMAT, graph.source(edge), graph.target(edge));
                }
            };
        }
        return new Function<Object, String>(){

            @Override
            public String apply(Object edge) {
                return graph.incidentNodes(edge).toString();
            }
        };
    }

    private static String getPropertiesString(Graph<?> graph) {
        if (graph instanceof Network) {
            return Graphs.getPropertiesString((Network)graph);
        }
        return String.format("isDirected: %s, allowsSelfLoops: %s", graph.isDirected(), graph.allowsSelfLoops());
    }

    private static String getPropertiesString(Network<?, ?> graph) {
        return String.format("isDirected: %s, allowsParallelEdges: %s, allowsSelfLoops: %s", graph.isDirected(), graph.allowsParallelEdges(), graph.allowsSelfLoops());
    }
}

