/*
 * Copyright 2010 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.drools.core.rule;

import java.util.Collection;

import org.drools.core.common.EventFactHandle;
import org.drools.core.common.InternalFactHandle;
import org.drools.core.common.InternalWorkingMemory;
import org.drools.core.spi.PropagationContext;
import org.drools.core.spi.RuleComponent;
import org.drools.core.time.JobHandle;

/**
 * An interface for all behavior implementations
 */
public interface Behavior extends RuleComponent, Cloneable {

    interface Context {
        Collection<EventFactHandle> getFactHandles();

        default JobHandle getJobHandle() {
            return null;
        }

        default void setJobHandle(JobHandle jobHandle) {
            throw new UnsupportedOperationException();
        }
    }
    
    enum BehaviorType {
        TIME_WINDOW( "time" ),
        LENGTH_WINDOW( "length" );
        
        private final String id;
        
        BehaviorType( String id ) {
            this.id = id;
        }
        
        public boolean matches( String id ) {
            return this.id.equalsIgnoreCase( id );
        }
    }
    
    /**
     * Returns the type of the behavior
     */
    BehaviorType getType();

    /**
     * Creates the context object associated with this behavior.
     * The object is given as a parameter in all behavior call backs.
     */
    Context createContext();

    /**
     * Makes the behavior aware of the new fact entering behavior's scope
     * 
     * @param context The behavior context object
     * @param fact The new fact entering behavior's scope
     * @param workingMemory The working memory session reference
     * 
     * @return true if the propagation should continue, false otherwise. I.e., 
     *         the behaviour has veto power over the fact propagation, and prevents
     *         the propagation to continue if returns false on this method. 
     */
    boolean assertFact(Object context,
                              InternalFactHandle fact,
                              PropagationContext pctx,
                              InternalWorkingMemory workingMemory);

    /**
     * Removes a right tuple from the behavior's scope
     * 
     * @param context The behavior context object
     * @param fact The fact leaving the behavior's scope
     * @param workingMemory The working memory session reference
     */
    void retractFact(Object context,
                            InternalFactHandle fact,
                            PropagationContext pctx,
                            InternalWorkingMemory workingMemory);

    /**
     * A callback method that allows behaviors to expire facts
     */
    void expireFacts(Object context,
                            PropagationContext pctx,
                            InternalWorkingMemory workingMemory);

    /**
     * Some behaviors might change the expiration offset for the 
     * associated fact type. Example: time sliding windows. 
     * 
     * For these behaviors, this method must return the expiration
     * offset associated to them.
     * 
     * @return the expiration offset for this behavior or -1 if 
     *         they don't have a time based expiration offset.
     */
    long getExpirationOffset();
    
}
