/*
 * Decompiled with CFR 0.152.
 */
package org.kie.hacep.core.infra.election;

import io.fabric8.kubernetes.api.model.ConfigMap;
import io.fabric8.kubernetes.api.model.PodList;
import io.fabric8.kubernetes.client.KubernetesClient;
import io.fabric8.kubernetes.client.dsl.FilterWatchListDeletable;
import io.fabric8.kubernetes.client.dsl.NonNamespaceOperation;
import io.fabric8.kubernetes.client.dsl.Replaceable;
import io.fabric8.kubernetes.client.dsl.Resource;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.kie.hacep.core.GlobalStatus;
import org.kie.hacep.core.infra.election.ConfigMapLockUtils;
import org.kie.hacep.core.infra.election.KubernetesLockConfiguration;
import org.kie.hacep.core.infra.election.LeaderElection;
import org.kie.hacep.core.infra.election.LeaderInfo;
import org.kie.hacep.core.infra.election.LeadershipCallback;
import org.kie.hacep.core.infra.election.State;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LeaderElectionImpl
implements LeaderElection {
    private static final Logger logger = LoggerFactory.getLogger(LeaderElectionImpl.class);
    private KubernetesClient kubernetesClient;
    private KubernetesLockConfiguration lockConfiguration;
    private State currentState = State.REPLICA;
    private ScheduledExecutorService serializedExecutor;
    private volatile LeaderInfo latestLeaderInfo;
    private volatile ConfigMap latestConfigMap;
    private volatile Set<String> latestMembers;
    private List<LeadershipCallback> callbacks;

    public LeaderElectionImpl(KubernetesClient kubernetesClient, KubernetesLockConfiguration lockConfiguration, State initialState) {
        this.kubernetesClient = kubernetesClient;
        this.lockConfiguration = lockConfiguration;
        this.callbacks = new ArrayList<LeadershipCallback>();
        if (initialState != null) {
            this.currentState = initialState;
        }
    }

    @Override
    public void start() {
        if (this.serializedExecutor == null) {
            if (logger.isDebugEnabled()) {
                logger.debug("{} Starting leadership election...", (Object)this.logPrefix());
            }
            this.serializedExecutor = Executors.newSingleThreadScheduledExecutor();
            this.serializedExecutor.execute(this::refreshStatus);
        }
    }

    @Override
    public void stop() {
        if (logger.isDebugEnabled()) {
            logger.debug("{} Stopping leadership election...", (Object)this.logPrefix());
        }
        if (this.serializedExecutor != null) {
            this.serializedExecutor.shutdownNow();
        }
        this.serializedExecutor = null;
    }

    @Override
    public void addCallbacks(List<LeadershipCallback> callbacks) {
        this.callbacks.addAll(callbacks);
    }

    public void refreshStatus() {
        switch (this.currentState) {
            case REPLICA: {
                this.refreshStatusNotLeader();
                break;
            }
            case BECOMING_LEADER: {
                this.refreshStatusBecomingLeader();
                break;
            }
            case LEADER: {
                this.refreshStatusLeader();
                break;
            }
            default: {
                throw new IllegalStateException("Unsupported state " + this.currentState);
            }
        }
        for (LeadershipCallback callback : this.callbacks) {
            callback.updateStatus(this.currentState);
        }
    }

    void refreshStatusNotLeader() {
        boolean pulled;
        if (logger.isDebugEnabled()) {
            logger.debug("{} Pod is not leader, pulling new data from the cluster", (Object)this.logPrefix());
        }
        if (!(pulled = this.lookupNewLeaderInfo())) {
            this.rescheduleAfterDelay();
            return;
        }
        if (this.latestLeaderInfo.hasEmptyLeader()) {
            boolean acquired;
            if (logger.isInfoEnabled()) {
                logger.info("{} The cluster has no leaders. Trying to acquire the leadership...", (Object)this.logPrefix());
            }
            if (acquired = this.tryAcquireLeadership()) {
                if (logger.isInfoEnabled()) {
                    logger.info("{} Leadership acquired by current pod with immediate effect", (Object)this.logPrefix());
                }
                this.currentState = State.LEADER;
                this.serializedExecutor.execute(this::refreshStatus);
                return;
            }
            if (logger.isInfoEnabled()) {
                logger.info("{} Unable to acquire the leadership, it may have been acquired by another pod", (Object)this.logPrefix());
            }
        } else {
            if (!GlobalStatus.canBecomeLeader()) {
                if (logger.isInfoEnabled()) {
                    logger.info("{} Pod is not initialized yet (waiting snapshot) so cannot try to become leader", (Object)this.logPrefix());
                }
                this.rescheduleAfterDelay();
                return;
            }
            if (!this.latestLeaderInfo.hasValidLeader()) {
                boolean acquired;
                if (logger.isInfoEnabled()) {
                    logger.info("{} Leadership has been lost by old owner. Trying to acquire the leadership...", (Object)this.logPrefix());
                }
                if (acquired = this.tryAcquireLeadership()) {
                    if (logger.isInfoEnabled()) {
                        logger.info("{} Leadership acquired by current pod", (Object)this.logPrefix());
                    }
                    this.currentState = State.BECOMING_LEADER;
                    this.serializedExecutor.execute(this::refreshStatus);
                    return;
                }
                if (logger.isInfoEnabled()) {
                    logger.info("{} Unable to acquire the leadership, it may have been acquired by another pod", (Object)this.logPrefix());
                }
            } else if (this.latestLeaderInfo.isValidLeader(this.lockConfiguration.getPodName())) {
                if (logger.isInfoEnabled()) {
                    logger.info("{} Leadership is already owned by current pod", (Object)this.logPrefix());
                }
                this.currentState = State.BECOMING_LEADER;
                this.serializedExecutor.execute(this::refreshStatus);
                return;
            }
        }
        this.rescheduleAfterDelay();
    }

    void refreshStatusBecomingLeader() {
        long delay = this.lockConfiguration.getLeaseDurationMillis();
        if (logger.isInfoEnabled()) {
            logger.info("{} Current pod owns the leadership, but it will be effective in {} seconds...", (Object)this.logPrefix(), (Object)new BigDecimal(delay).divide(BigDecimal.valueOf(1000L), 2, RoundingMode.HALF_UP));
        }
        try {
            Thread.sleep(delay);
        }
        catch (InterruptedException e) {
            logger.warn("Thread interrupted", (Throwable)e);
            Thread.currentThread().interrupt();
        }
        if (logger.isInfoEnabled()) {
            logger.info("{} Current pod is becoming the new LEADER now...", (Object)this.logPrefix());
        }
        this.currentState = State.LEADER;
        this.serializedExecutor.execute(this::refreshStatus);
    }

    void refreshStatusLeader() {
        boolean pulled;
        if (logger.isDebugEnabled()) {
            logger.debug("{} Pod should be the leader, pulling new data from the cluster", (Object)this.logPrefix());
        }
        if (!(pulled = this.lookupNewLeaderInfo())) {
            this.rescheduleAfterDelay();
            return;
        }
        if (this.latestLeaderInfo.isValidLeader(this.lockConfiguration.getPodName())) {
            if (logger.isDebugEnabled()) {
                logger.debug("{} Current Pod is still the leader", (Object)this.logPrefix());
            }
            this.rescheduleAfterDelay();
        } else {
            if (logger.isDebugEnabled()) {
                logger.debug("{} Current Pod has lost the leadership", (Object)this.logPrefix());
            }
            this.currentState = State.REPLICA;
            this.serializedExecutor.execute(this::refreshStatus);
        }
    }

    void rescheduleAfterDelay() {
        this.serializedExecutor.schedule(this::refreshStatus, this.jitter(this.lockConfiguration.getRetryPeriodMillis(), this.lockConfiguration.getJitterFactor()), TimeUnit.MILLISECONDS);
    }

    public boolean lookupNewLeaderInfo() {
        Set<String> members;
        ConfigMap configMap;
        if (logger.isDebugEnabled()) {
            logger.debug("{} Looking up leadership information...", (Object)this.logPrefix());
        }
        try {
            configMap = this.pullConfigMap();
        }
        catch (Exception e) {
            logger.warn("{} Unable to retrieve the current ConfigMap {} from Kubernetes", (Object)this.logPrefix(), (Object)this.lockConfiguration.getConfigMapName());
            logger.debug("{} Exception thrown during ConfigMap lookup", (Object)this.logPrefix(), (Object)e);
            return false;
        }
        try {
            members = Objects.requireNonNull(this.pullClusterMembers(), "Retrieved a null set of members");
        }
        catch (Exception e) {
            logger.warn("{} Unable to retrieve the list of cluster members from Kubernetes", (Object)this.logPrefix());
            logger.debug("{} Exception thrown during Pod list lookup ", (Object)this.logPrefix(), (Object)e);
            return false;
        }
        this.updateLatestLeaderInfo(configMap, members);
        return true;
    }

    boolean tryAcquireLeadership() {
        LeaderInfo leaderInfo;
        boolean canAcquire;
        if (logger.isDebugEnabled()) {
            logger.debug("{} Trying to acquire the leadership...", (Object)this.logPrefix());
        }
        ConfigMap configMap = this.latestConfigMap;
        Set<String> members = this.latestMembers;
        LeaderInfo latestLeaderInfoLocal = this.latestLeaderInfo;
        if (latestLeaderInfoLocal == null || members == null) {
            if (logger.isWarnEnabled()) {
                logger.warn("{} Unexpected condition. Latest leader info or list of members is empty.", (Object)this.logPrefix());
            }
            return false;
        }
        if (!members.contains(this.lockConfiguration.getPodName())) {
            if (logger.isWarnEnabled()) {
                logger.warn("{} The list of cluster members {} does not contain the current Pod. Cannot acquire leadership.", (Object)this.logPrefix(), latestLeaderInfoLocal.getMembers());
            }
            return false;
        }
        LeaderInfo newLeaderInfo = new LeaderInfo(this.lockConfiguration.getGroupName(), this.lockConfiguration.getPodName(), new Date(), members);
        if (configMap == null) {
            if (logger.isDebugEnabled()) {
                logger.debug("{} Lock configmap is not present in the Kubernetes namespace. A new ConfigMap will be created", (Object)this.logPrefix());
            }
            ConfigMap newConfigMap = ConfigMapLockUtils.createNewConfigMap(this.lockConfiguration.getConfigMapName(), newLeaderInfo);
            try {
                ((NonNamespaceOperation)this.kubernetesClient.configMaps().inNamespace(this.lockConfiguration.getKubernetesResourcesNamespaceOrDefault(this.kubernetesClient))).create((Object[])new ConfigMap[]{newConfigMap});
                if (logger.isDebugEnabled()) {
                    logger.debug("{} ConfigMap {} successfully created", (Object)this.logPrefix(), (Object)this.lockConfiguration.getConfigMapName());
                }
                this.updateLatestLeaderInfo(newConfigMap, members);
                return true;
            }
            catch (Exception ex) {
                logger.warn("Unable to create the ConfigMap, it may have been created by other cluster members concurrently. If the problem persists, check if the service account has the right permissions to create it", (Object)this.logPrefix());
                logger.debug("{} Exception while trying to create the ConfigMap", (Object)this.logPrefix(), (Object)ex);
                return false;
            }
        }
        if (logger.isDebugEnabled()) {
            logger.debug("{} Lock configmap already present in the Kubernetes namespace. Checking...", (Object)this.logPrefix());
        }
        boolean bl = canAcquire = !(leaderInfo = ConfigMapLockUtils.getLeaderInfo(configMap, members, this.lockConfiguration.getGroupName())).hasValidLeader();
        if (canAcquire) {
            try {
                ConfigMap updatedConfigMap = ConfigMapLockUtils.getConfigMapWithNewLeader(configMap, newLeaderInfo);
                ((Replaceable)((Resource)((NonNamespaceOperation)this.kubernetesClient.configMaps().inNamespace(this.lockConfiguration.getKubernetesResourcesNamespaceOrDefault(this.kubernetesClient))).withName(this.lockConfiguration.getConfigMapName())).lockResourceVersion(configMap.getMetadata().getResourceVersion())).replace((Object)updatedConfigMap);
                if (logger.isDebugEnabled()) {
                    logger.debug("{} ConfigMap {} successfully updated", (Object)this.logPrefix(), (Object)this.lockConfiguration.getConfigMapName());
                }
                this.updateLatestLeaderInfo(updatedConfigMap, members);
                return true;
            }
            catch (Exception ex) {
                logger.warn("{} Unable to update the lock ConfigMap to set leadership information", (Object)this.logPrefix());
                logger.debug("{} Error received during configmap lock replace", (Object)this.logPrefix(), (Object)ex);
                return false;
            }
        }
        if (logger.isDebugEnabled()) {
            logger.debug("{} Another Pod ({}) is the current leader and it is still active", (Object)this.logPrefix(), (Object)this.latestLeaderInfo.getLeader());
        }
        return false;
    }

    void updateLatestLeaderInfo(ConfigMap configMap, Set<String> members) {
        if (logger.isDebugEnabled()) {
            logger.debug("{} Updating internal status about the current leader", (Object)this.logPrefix());
        }
        this.latestConfigMap = configMap;
        this.latestMembers = members;
        this.latestLeaderInfo = ConfigMapLockUtils.getLeaderInfo(configMap, members, this.lockConfiguration.getGroupName());
        if (logger.isDebugEnabled()) {
            logger.debug("{} Current leader info: {}", (Object)this.logPrefix(), (Object)this.latestLeaderInfo);
        }
    }

    ConfigMap pullConfigMap() {
        return (ConfigMap)((Resource)((NonNamespaceOperation)this.kubernetesClient.configMaps().inNamespace(this.lockConfiguration.getKubernetesResourcesNamespaceOrDefault(this.kubernetesClient))).withName(this.lockConfiguration.getConfigMapName())).get();
    }

    public Set<String> pullClusterMembers() {
        List pods = ((PodList)((FilterWatchListDeletable)((NonNamespaceOperation)this.kubernetesClient.pods().inNamespace(this.lockConfiguration.getKubernetesResourcesNamespaceOrDefault(this.kubernetesClient))).withLabels(this.lockConfiguration.getClusterLabels())).list()).getItems();
        return pods.stream().map(pod -> pod.getMetadata().getName()).collect(Collectors.toSet());
    }

    private long jitter(long num, double factor) {
        return (long)((double)num * (1.0 + Math.random() * (factor - 1.0)));
    }

    private String logPrefix() {
        return "Pod[" + this.lockConfiguration.getPodName() + "]";
    }
}

