/*
 * Decompiled with CFR 0.152.
 */
package org.kie.processmigration.service.impl;

import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import javax.enterprise.context.ApplicationScoped;
import javax.inject.Inject;
import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.PersistenceContext;
import javax.persistence.TypedQuery;
import javax.transaction.Transactional;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Entity;
import javax.ws.rs.core.Response;
import org.apache.commons.lang3.StringUtils;
import org.kie.processmigration.model.Execution;
import org.kie.processmigration.model.Migration;
import org.kie.processmigration.model.MigrationDefinition;
import org.kie.processmigration.model.MigrationReport;
import org.kie.processmigration.model.Plan;
import org.kie.processmigration.model.exceptions.InvalidKieServerException;
import org.kie.processmigration.model.exceptions.InvalidMigrationException;
import org.kie.processmigration.model.exceptions.MigrationNotFoundException;
import org.kie.processmigration.model.exceptions.PlanNotFoundException;
import org.kie.processmigration.model.exceptions.ProcessNotFoundException;
import org.kie.processmigration.model.exceptions.ReScheduleException;
import org.kie.processmigration.service.KieService;
import org.kie.processmigration.service.MigrationService;
import org.kie.processmigration.service.PlanService;
import org.kie.processmigration.service.SchedulerService;
import org.kie.processmigration.service.TransactionHelper;
import org.kie.server.api.model.admin.MigrationReportInstance;
import org.kie.server.api.model.instance.ProcessInstance;
import org.kie.server.client.QueryServicesClient;
import org.kie.server.client.admin.ProcessAdminServicesClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ApplicationScoped
public class MigrationServiceImpl
implements MigrationService {
    private static final Logger logger = LoggerFactory.getLogger(MigrationServiceImpl.class);
    private static final List<Integer> QUERY_PROCESS_INSTANCE_STATUSES = Arrays.asList(1);
    public static final Integer QUERY_PAGE_SIZE = 100;
    @PersistenceContext
    private EntityManager em;
    @Inject
    private PlanService planService;
    @Inject
    private KieService kieService;
    @Inject
    private SchedulerService schedulerService;
    @Inject
    private TransactionHelper txHelper;

    public Migration get(Long id) throws MigrationNotFoundException {
        TypedQuery query = this.em.createNamedQuery("Migration.findById", Migration.class);
        query.setParameter("id", (Object)id);
        try {
            return (Migration)query.getSingleResult();
        }
        catch (NoResultException e) {
            throw new MigrationNotFoundException(id);
        }
    }

    public List<MigrationReport> getResults(Long id) throws MigrationNotFoundException {
        Migration m = this.get(id);
        TypedQuery query = this.em.createNamedQuery("MigrationReport.findByMigrationId", MigrationReport.class);
        query.setParameter("id", (Object)m.getId());
        return query.getResultList();
    }

    public List<Migration> findAll() {
        return this.em.createNamedQuery("Migration.findAll", Migration.class).getResultList();
    }

    public Migration submit(MigrationDefinition definition) throws InvalidMigrationException {
        this.validateDefinition(definition);
        Migration migration = (Migration)this.txHelper.withTransaction(() -> {
            Migration m = new Migration(definition);
            this.em.persist((Object)m);
            return m;
        });
        if (Execution.ExecutionType.SYNC.equals((Object)definition.getExecution().getType())) {
            this.migrate(migration);
        } else {
            this.validatePlanExecution(definition);
            this.schedulerService.scheduleMigration(migration);
        }
        return migration;
    }

    @Transactional
    public Migration delete(Long id) throws MigrationNotFoundException {
        Migration migration = this.get(id);
        this.em.remove((Object)migration);
        return migration;
    }

    public Migration update(Long id, MigrationDefinition definition) throws MigrationNotFoundException, ReScheduleException, InvalidMigrationException {
        this.validateDefinition(definition);
        Migration migration = this.get(id);
        if (!Execution.ExecutionStatus.SCHEDULED.equals((Object)migration.getStatus())) {
            throw new ReScheduleException("The migration is not scheduled and cannot be re-scheduled");
        }
        if (Execution.ExecutionType.SYNC.equals((Object)definition.getExecution().getType())) {
            throw new ReScheduleException("The migration execution type MUST be ASYNC");
        }
        migration.setDefinition(definition);
        this.txHelper.withTransaction(() -> (Migration)this.em.merge((Object)migration));
        this.schedulerService.reScheduleMigration(migration);
        return migration;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Migration migrate(Migration migration) throws InvalidMigrationException {
        try {
            Plan plan = this.planService.get(migration.getDefinition().getPlanId());
            this.validatePlanExecution(migration.getDefinition(), plan);
            if (Execution.ExecutionStatus.CREATED.equals((Object)migration.getStatus()) || Execution.ExecutionStatus.SCHEDULED.equals((Object)migration.getStatus())) {
                migration.start();
            }
            AtomicBoolean hasErrors = new AtomicBoolean(false);
            List instanceIds = this.getInstancesToMigrate(migration);
            ProcessAdminServicesClient adminService = this.kieService.getProcessAdminServicesClient(migration.getDefinition().getKieServerId());
            QueryServicesClient queryService = this.kieService.getQueryServicesClient(migration.getDefinition().getKieServerId());
            for (Long instanceId : instanceIds) {
                boolean successful = this.migrateInstance(instanceId, migration, plan, adminService, queryService);
                if (hasErrors.get() || Boolean.TRUE.equals(successful)) continue;
                hasErrors.set(Boolean.TRUE);
            }
            migration.complete(Boolean.valueOf(hasErrors.get()));
        }
        catch (ProcessNotFoundException e) {
            migration.fail((Exception)((Object)e));
            throw e;
        }
        catch (PlanNotFoundException e) {
            migration.fail((Exception)((Object)e));
            throw new InvalidMigrationException("The provided plan id does not exist: " + migration.getDefinition().getPlanId());
        }
        catch (Exception e) {
            logger.warn("Migration failed", (Throwable)e);
            migration.fail(e);
        }
        finally {
            this.txHelper.withTransaction(() -> (Migration)this.em.merge((Object)migration));
            if (Execution.ExecutionType.ASYNC.equals((Object)migration.getDefinition().getExecution().getType()) && migration.getDefinition().getExecution().getCallbackUrl() != null) {
                this.doCallback(migration);
            }
        }
        return migration;
    }

    private boolean migrateInstance(Long instanceId, Migration migration, Plan plan, ProcessAdminServicesClient adminService, QueryServicesClient queryService) {
        MigrationReportInstance reportInstance = null;
        try {
            ProcessInstance pi = queryService.findProcessInstanceById(instanceId);
            if (pi != null && pi.getContainerId().equals(plan.getSource().getContainerId())) {
                reportInstance = adminService.migrateProcessInstance(plan.getSource().getContainerId(), instanceId, plan.getTarget().getContainerId(), plan.getTarget().getProcessId(), plan.getMappings());
            } else {
                reportInstance = this.buildReport(instanceId);
                reportInstance.setLogs(Arrays.asList("Instance did not exist in source container. Migration skipped"));
                logger.debug("Process Instance {} did not exist in source container with id {}", (Object)instanceId, (Object)plan.getSource().getContainerId());
            }
        }
        catch (Exception e) {
            logger.warn("Unable to migrate instanceID: " + instanceId, (Throwable)e);
            reportInstance = this.buildReportFromError(instanceId, e);
        }
        MigrationReport report = new MigrationReport(migration.getId(), reportInstance);
        return ((MigrationReport)this.txHelper.withTransaction(() -> {
            this.em.persist((Object)report);
            return report;
        })).getSuccessful();
    }

    private void doCallback(Migration migration) {
        URI callbackURI = null;
        try {
            callbackURI = migration.getDefinition().getExecution().getCallbackUrl();
            Response response = ClientBuilder.newClient().target(callbackURI).request(new String[]{"application/json"}).buildPost(Entity.json((Object)migration)).invoke();
            if (Response.Status.OK.getStatusCode() == response.getStatus()) {
                logger.debug("Migration [{}] - Callback to {} replied successfully", (Object)migration.getId(), (Object)callbackURI);
            } else {
                logger.warn("Migration [{}] - Callback to {} replied with {}", new Object[]{migration.getId(), callbackURI, response.getStatus()});
            }
        }
        catch (Exception e) {
            logger.error("Migration [{}] - Callback to {} failed.", new Object[]{migration.getId(), callbackURI, e});
        }
    }

    public void validateDefinition(MigrationDefinition definition) throws InvalidMigrationException {
        if (definition == null) {
            throw new InvalidMigrationException("The Migration Definition must not be null");
        }
        if (definition.getPlanId() == null) {
            throw new InvalidMigrationException("The Plan ID is mandatory");
        }
        if (StringUtils.isBlank((CharSequence)definition.getKieServerId())) {
            throw new InvalidMigrationException("The KIE Server ID is mandatory");
        }
        if (!this.kieService.hasKieServer(definition.getKieServerId())) {
            throw new InvalidKieServerException(definition.getKieServerId());
        }
    }

    private void validatePlanExecution(MigrationDefinition definition) throws InvalidMigrationException {
        try {
            this.validatePlanExecution(definition, this.planService.get(definition.getPlanId()));
        }
        catch (PlanNotFoundException e) {
            throw new InvalidMigrationException("Plan not found with ID: " + definition.getPlanId());
        }
    }

    private void validatePlanExecution(MigrationDefinition definition, Plan plan) throws InvalidMigrationException {
        if (!this.kieService.existsProcessDefinition(definition.getKieServerId(), plan.getSource())) {
            throw new ProcessNotFoundException(plan.getSource().getContainerId());
        }
        if (!this.kieService.existsProcessDefinition(definition.getKieServerId(), plan.getTarget())) {
            throw new ProcessNotFoundException(plan.getTarget().getContainerId());
        }
    }

    private List<Long> getInstancesToMigrate(Migration migration) throws InvalidKieServerException, PlanNotFoundException {
        List instanceIds = migration.getDefinition().getProcessInstanceIds();
        ArrayList migratedInstances = new ArrayList();
        if (migration.getReports() != null && !migration.getReports().isEmpty()) {
            migration.getReports().stream().map(r -> r.getProcessInstanceId()).forEach(id -> migratedInstances.add(id));
        }
        Plan plan = this.planService.get(migration.getDefinition().getPlanId());
        if (instanceIds == null || instanceIds.isEmpty()) {
            boolean allFetched = false;
            int page = 0;
            while (!allFetched) {
                List instances = this.kieService.getQueryServicesClient(migration.getDefinition().getKieServerId()).findProcessInstancesByContainerId(plan.getSource().getContainerId(), QUERY_PROCESS_INSTANCE_STATUSES, Integer.valueOf(page++), QUERY_PAGE_SIZE);
                instances.stream().forEach(p -> instanceIds.add(p.getId()));
                if (instances.size() >= QUERY_PAGE_SIZE) continue;
                allFetched = true;
            }
        }
        return instanceIds.stream().filter(id -> !migratedInstances.contains(id)).collect(Collectors.toList());
    }

    private MigrationReportInstance buildReport(Long instanceId) {
        MigrationReportInstance reportInstance = new MigrationReportInstance();
        reportInstance.setSuccessful(true);
        reportInstance.setProcessInstanceId(instanceId);
        reportInstance.setStartDate(new Date());
        reportInstance.setEndDate(new Date());
        return reportInstance;
    }

    private MigrationReportInstance buildReportFromError(Long instanceId, Exception e) {
        MigrationReportInstance reportInstance = this.buildReport(instanceId);
        reportInstance.setSuccessful(false);
        reportInstance.setLogs(Arrays.asList(e.getMessage()));
        return reportInstance;
    }
}

