/**
 * $Id: UserLookupProvider.java 3258 2008-10-30 17:40:54Z azeckoski $
 * $URL: https://scm.dspace.org/svn/repo/dspace2/core/tags/OR09/api/src/main/java/org/dspace/providers/UserLookupProvider.java $
 * UserLookupService.java - DSpace2 - Oct 13, 2008 11:51:15 AM - azeckoski
 **************************************************************************
 * Copyright (c) 2008 Aaron Zeckoski
 * Licensed under the Apache License, Version 2.0
 * 
 * A copy of the Apache License has been included in this 
 * distribution and is available at: http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Aaron Zeckoski (azeckoski @ gmail.com) (aaronz @ vt.edu) (aaron @ caret.cam.ac.uk)
 */

package org.dspace.providers;

import java.util.List;
import java.util.Map;

import org.dspace.services.model.User;
import org.dspace.services.model.UserSearch;

/**
 * This allows customization of the user lookup system for DSpace by allowing
 * an external system to look up user information and search for users <br/>
 * Note that all user lookups will be cached and that cache controlled by the
 * system administrator such that the developer should not be concerned about doing many
 * user lookups or many calls to the methods in this provider <br/>
 * NOTE about external identifiers: <br/>
 * The external id (eid) is the identifier that is used in the source for your user data
 * and is supplied by you when the user is first authenticated, once this id is known it will
 * be used for lookups of data in this provider,
 * if you are not providing the user eid then it will be the username (login name) of the user
 * 
 * @author Aaron Zeckoski (azeckoski @ gmail.com)
 */
public interface UserLookupProvider {

    /**
     * Get a populated user object for the given userEid,
     * this is the main method for this provider and will be called most often,
     * see the note in this class for more info about Eids
     * 
     * @param userEid the user external identifier (often the username, cannot be null or blank)
     * @return a populated {@link User} object OR null if this is not a valid user eid
     */
    public User getUserByEid(String userEid);

    /**
     * Get user objects populated with data based on the input set of userEids,
     * should return one {@link User} object per input EID and in the
     * same order as the inputs, if any users are invalid then
     * a null should be returned in the position for that user object <br/>
     * This is the efficient multiple inputs version of {@link #getUserById(String)},
     * if you have no way to implement this more efficiently than single lookups then
     * simply call the {@link #getUserByEid(String)} method in a loop <br/>
     * 
     * @param userEids an array of the external user ids for users
     * @return a map of userId to {@link User} objects which match with the input ids
     */
    public Map<String, User> getUsersByEids(String[] userEids);

    /**
     * Get user objects populated with data based on a set of search parameters,
     * will return an empty list if no matches are found for this search,
     * you should respect searches by user eid, surname, and email at least <br/>
     * 
     * @param search a search object containing the data about the search
     * including limits, fields, and ordering
     * @return a list of users based on the given search
     * @throws IllegalArgumentException if the search is null
     */
    public List<User> getUsersBySearch(UserSearch search);

}
