/*
 * $URL: https://dspace.svn.sf.net/svnroot/dspace/dspace2/trunk/api/src/main/java/org/dspace/services/exceptions/AuthorizeException.java $
 * 
 * $Revision: 3173 $
 * 
 * $Date: 2008-10-09 07:49:25 -0700 (Thu, 09 Oct 2008) $
 *
 * Copyright (c) 2008, The DSpace Foundation.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * - Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * - Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 * - Neither the name of the DSpace Foundation nor the names of its 
 * contributors may be used to endorse or promote products derived from 
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
package org.dspace.services;

import org.dspace.services.model.MetadataSchema;

/**
 * Class representing a schema in DSpace.
 * <p>
 * The schema object exposes a name which can later be used to generate
 * namespace prefixes in RDF or XML, e.g. the core DSpace Dublin Core schema
 * would have a name of <code>'dc'</code>.
 * </p>
 *
 * @author Martin Hald
 * @version $Revision: 3036 $
 */
public interface MetadataSchemaService
{
	
    /** Numeric Identifier of built-in Dublin Core schema. */
    public static final int DC_SCHEMA_ID = 1;

    /** Short Name of built-in Dublin Core schema. */
    public static final String DC_SCHEMA = "dc";

    public static final String SCHEMA_KEY = "schema";
    public static final String SCHEMA_REF_PREFIX = "/"+SCHEMA_KEY+"/";
    /**
	 * Creates a new metadata schema in the database, out of this object.
	 * @param schema to create
	 */
	public void create(MetadataSchema schema);

	/**
	 * Update the metadata schema in the database.
	 * @param schema to update
	 */
	public MetadataSchema update(MetadataSchema schema);

	/**
	 * Delete the metadata schema.
	 * @param schema to delete
	 */
	public boolean delete(MetadataSchema schema);

    /**
     * Get the schema object corresponding to this namespace URI.
     * @param namespace namespace URI to match
     * @return metadata schema object or null if none found.
     */
    public MetadataSchema findByNamespace(String namespace);

    /**
     * Return all metadata schemas.
     * @return array of metadata schemas
     */
    public MetadataSchema[] findAll();

    /**
     * Get the schema corresponding with this numeric ID.
     * The ID is a database key internal to DSpace.
     * @param id the schema ID
     * @return the metadata schema object
     */
    public MetadataSchema findById( String id);

    /**
     * Get the schema corresponding with this short name.
     * @param prefix the short name for the schema
     * @return the metadata schema object
     */
    public MetadataSchema findByPrefix(String prefix);
    
}
