/**
 * $Id: SimpleStorageService.java 3203 2008-10-14 16:54:39Z azeckoski $
 * $URL: https://scm.dspace.org/svn/repo/dspace2/core/tags/OR09/api/src/main/java/org/dspace/services/SimpleStorageService.java $
 * SimpleStorageService.java - DSpace2 - Oct 9, 2008 3:58:32 PM - azeckoski
 **************************************************************************
 * Copyright (c) 2008 Aaron Zeckoski
 * Licensed under the Apache License, Version 2.0
 * 
 * A copy of the Apache License has been included in this 
 * distribution and is available at: http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Aaron Zeckoski (azeckoski @ gmail.com) (aaronz @ vt.edu) (aaron @ caret.cam.ac.uk)
 */

package org.dspace.services;

/**
 * A simple service to store data persistently,
 * the location of the stored data is controlled by the implementation of this service
 * but it would probably be stored in the repository itself
 * 
 * @author Aaron Zeckoski (azeckoski @ gmail.com)
 */
public interface SimpleStorageService {

    /**
     * Stores a value under a key and optionally associates it with with a user,
     * if the key is already used then the value is updated for that key,
     * if the value is null then storage for that key is deleted <br/>
     * admins or internal processes can leave the userId blank if desired to have it associated with the core system
     * or can write any data to be associated with any user
     * 
     * @param userId (optional) the internal user id for a user (not the username or EID),
     * if null this will be associated with the current user
     * @param key a unique key which can be used to retrieve the stored data
     * @param value the data to store
     * @throws IllegalArgumentException if the key is null or there is no current user
     * @throws SecurityException if the current user does not have permission to write to this key
     */
    public void store(String userId, String key, String value);

    /**
     * Retrieve stored data if it exists for the given user <br/>
     * 
     * @param userId (optional) the internal user id for a user (not the username or EID),
     * if null this will use the current user
     * @param key a unique key which can be used to retrieve the stored data
     * @return the stored value for this key OR null if none exists
     * @throws IllegalArgumentException if the data cannot be converted to a string
     * @throws SecurityException if the current user does not have permission to read this key
     */
    public String retrieve(String userId, String key);

}
