/**
 * $Id: StorageService.java 3670 2009-03-31 22:53:04Z mdiggory $
 * $URL: https://scm.dspace.org/svn/repo/dspace2/core/tags/OR09/api/src/main/java/org/dspace/services/StorageService.java $
 * StorageService2.java - DS2 - Feb 2, 2009 10:30:34 AM - azeckoski
 **************************************************************************
 * Copyright (c) 2008 Aaron Zeckoski
 * Licensed under the Apache License, Version 2.0
 * 
 * A copy of the Apache License has been included in this 
 * distribution and is available at: http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Aaron Zeckoski (azeckoski @ gmail.com) (aaronz @ vt.edu) (aaron @ caret.cam.ac.uk)
 */

package org.dspace.services;

import java.util.List;

import org.dspace.services.exceptions.StorageInvalidPathException;
import org.dspace.services.model.StorageEntity;

/**
 * This is an attempt to create a storage service which abstracts simplistic file/folder
 * storage with metadata associated,
 * a storage reference will look like so: /path/to/folder/name or path-to-folder:name or whatever <br/>
 * All operations in this interface are expected to be part of a request so they will become persistent at the end
 * of the request only
 * 
 * @author Aaron Zeckoski (azeckoski @ gmail.com)
 */
public interface StorageService {

    /**
     * This is the root path for the storage system
     */
    public static final String ROOT_PATH = "/";

    /**
     * Check to see if a storage entity exists at a given reference 
     * (should be faster than doing the retrieval)
     * 
     * @param reference the unique reference (path + name) of the given entity (e.g. trees/apple/golden_delicious)
     * @return true if there is a storage entity with this reference OR false if none can be found
     * @throws IllegalArgumentException if the reference is null or invalid
     */
    public boolean exists(String reference);

    /**
     * Retrieve an entity by a given reference
     * 
     * @param reference the unique reference (path + name) of the given entity (e.g. trees/apple/golden_delicious)
     * @return the entity object OR null if none can be found at the given reference
     * @throws IllegalArgumentException if the storageEntity is invalid
     */
    public StorageEntity getEntity(String reference);

    /**
     * Get a list of all storage entities at a given path, 
     * only includes children at the given path and will not recursively include multiple levels of children
     * 
     * @param path a path to a storage folder (e.g. trees/apple or path/to/storage)
     * @return a list of all storage entities at the given path
     * @throws StorageInvalidPathException if the path is invalid
     */
    public List<StorageEntity> getEntities(String path);

    /**
     * Check to see if a storage entity exists with a given identifier
     *
     * @param identifier the unique id of the given entity (e.g. the UUID)
     * @return true if there is a storage entity with this identifier OR false if none can be found
     * @throws IllegalArgumentException if the reference is null
     */
    public boolean existsById(String identifier);

    /**
     * Retrieve an entity by a given identifier
     *
     * @param identifier the unique id of the given entity (e.g. the UUID)
     * @return the entity object OR null if none can be found with this identifier
     * @throws IllegalArgumentException if the storageEntity is invalid
     */
    public StorageEntity getEntityById(String identifier);

}
