/**
 * $Id: StorageSearch.java 3442 2009-02-06 15:18:11Z azeckoski $
 * $URL: https://scm.dspace.org/svn/repo/dspace2/core/tags/OR09/api/src/main/java/org/dspace/services/mixins/StorageSearch.java $
 * UserSearch.java - DSpace2 - Oct 13, 2008 12:24:15 PM - azeckoski
 **************************************************************************
 * Copyright (c) 2008 Aaron Zeckoski
 * Licensed under the Apache License, Version 2.0
 * 
 * A copy of the Apache License has been included in this 
 * distribution and is available at: http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Aaron Zeckoski (azeckoski @ gmail.com) (aaronz @ vt.edu) (aaron @ caret.cam.ac.uk)
 */

package org.dspace.services.mixins;

import java.util.EnumMap;
import java.util.Map;

import org.dspace.search.Order;
import org.dspace.search.Restriction;
import org.dspace.search.Search;
import org.dspace.services.model.User;


/**
 * Allows for searching for storage entities and restricting the returned data,
 * Includes extensions which define the searchable storage identifier fields {@link SearchFields}
 * and some methods to make storage entity searching easier: 
 * {@link #addIdentifierSearch(SearchFields, String, int)} and {@link #addIdentifierOrder(SearchFields, boolean)}
 * The rest of the searching fields will be against the properties on 
 * 
 * @author Aaron Zeckoski (azeckoski @ gmail.com)
 */
public class StorageSearch extends Search {

    /**
     * Defines the searchable fields for the {@link User} object
     */
    public static enum SearchFields {
        /**
         *  the path to a given entity (e.g. trees/apple/golden_delicious, path=trees/apple),
         *  limits the search to a given path
         */
        PATH, 
        /**
         *  the name of a given entity (e.g. trees/apple/golden_delicious, name=golden_delicious),
         *  limits the search to entities with the given name
         */
        NAME
    };
    /**
     * Names in this map are searchable storage identifiers, 
     * all other search properties will be applied to searching against the
     * properties on storage entities,
     * the map goes from the search field enum -> the User object property name
     */
    public final static Map<SearchFields, String> searchPropertyMap;
    static {
        searchPropertyMap = new EnumMap<SearchFields, String>(SearchFields.class);
        searchPropertyMap.put(SearchFields.PATH, "path");
        searchPropertyMap.put(SearchFields.NAME, "name");
    }

    /**
     * Adds a single storage identifier search term
     * 
     * @param field the storage identifier field to search
     * @param searchValue the value to search for
     * @param comparison the comparison to make between the property and the value,
     * use the defined constants: e.g. EQUALS, LIKE, etc... from {@link Restriction}
     */
    public void addIdentifierSearch(SearchFields field, String searchValue, int comparison) {
        this.addRestriction( new Restriction(searchPropertyMap.get(field), searchValue, comparison) );
    }

    /**
     * Adds an ordering for the storage identifiers search
     * 
     * @param field the storage identifier field to search
     * @param ascending if true then the order will be ascending (natural order), otherwise descending
     */
    public void addIdentifierOrder(SearchFields field, boolean ascending) {
        this.addOrder( new Order(searchPropertyMap.get(field), ascending) );
    }

}
