/**
 * $Id: StorageVersionableWriteable.java 3443 2009-02-06 15:38:18Z azeckoski $
 * $URL: https://scm.dspace.org/svn/repo/dspace2/core/tags/OR09/api/src/main/java/org/dspace/services/mixins/StorageVersionableWriteable.java $
 * StorageMovable.java - DS2 - Feb 6, 2009 10:54:41 AM - azeckoski
 **************************************************************************
 * Copyright (c) 2008 Aaron Zeckoski
 * Licensed under the Apache License, Version 2.0
 * 
 * A copy of the Apache License has been included in this 
 * distribution and is available at: http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Aaron Zeckoski (azeckoski @ gmail.com) (aaronz @ vt.edu) (aaron @ caret.cam.ac.uk)
 */

package org.dspace.services.mixins;

import org.dspace.services.exceptions.StorageInvalidVersionException;
import org.dspace.services.model.StorageEntity;

/**
 * Allows storage entities to be retrieved according to their versions and allows
 * versioning information to be retrieved about entities,
 * also allows versions to be restored and versioning information to be written <br/>
 * One of the mixins for the storage service
 * 
 * @author Aaron Zeckoski (azeckoski @ gmail.com)
 */
public interface StorageVersionableWriteable extends StorageWriteable, StorageVersionable {

    /**
     * Restore a storage entity to a previous version, 
     * this will create a new current version which matches the given version,
     * if the given version is the current version then nothing happens <br/>
     * NOTE: this will automatically apply a label to the current version which includes information about the restored version
     * 
     * @param reference the unique reference (path + name) of the given entity (e.g. trees/apple/golden_delicious)
     * @param versionName the version key (normally obtained from the list in {@link #getVersions(String)})
     * @param path the path to place the restored version
     * @return the storage entity which represents the new current version (the restored one)
     * @throws IllegalArgumentException if any of the required arguments are null or invalid
     * @throws StorageInvalidVersionException if the given version does not exist
     * @throws StorageException if there is an internal storage system failure
     */
    public StorageEntity restoreVersion(String reference, String versionName, String label);

    /**
     * Set the version label, this is additional free text which is associated with a version specifically and
     * can be displayed as part of the information about this version,
     * this will overwrite any existing label text for this version <br/>
     * NOTE: attempting to set a version label on an entity with no versions will have no effect
     * 
     * @param reference the unique reference (path + name) of the given entity (e.g. trees/apple/golden_delicious)
     * @param versionName the version key (normally obtained from the list in {@link #getVersions(String)}),
     * using {@link #CURRENT_VERSION} will always retrieve the current version of the entity even if there
     * are no versions stored yet (this is the same as simply getting the entity)
     * @throws IllegalArgumentException if any of the required arguments are null or invalid
     * @throws StorageInvalidReferenceException if the given reference does not exist
     * @throws StorageInvalidVersionException if the given version does not exist
     * @throws StorageException if there is an internal storage system failure
     */
    public void setVersionLabel(String reference, String versionName, String label);

}
