/**
 * $Id: UserAuthorizationService.java 3194 2008-10-14 10:32:57Z azeckoski $
 * $URL: https://scm.dspace.org/svn/repo/dspace2/core/tags/OR09/api/src/main/java/org/dspace/services/mixins/UserAuthorizationService.java $
 * UserAuthorizationService.java - DSpace2 - Oct 14, 2008 10:53:14 AM - azeckoski
 **************************************************************************
 * Copyright (c) 2008 Aaron Zeckoski
 * Licensed under the Apache License, Version 2.0
 * 
 * A copy of the Apache License has been included in this 
 * distribution and is available at: http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Aaron Zeckoski (azeckoski @ gmail.com) (aaronz @ vt.edu) (aaron @ caret.cam.ac.uk)
 */

package org.dspace.services.mixins;

import java.util.List;

/**
 * This is the base authorization service which allows for checking permissions
 * for users on various things in the system as represented by their complete reference
 * (a string identifier which is unique for the object) <br/>
 * Deals in internal user IDs for all checks <br/>
 * This may be implemented by an external system to DSpace to allow greater flexibility
 * 
 * @author Aaron Zeckoski (azeckoski @ gmail.com)
 */
public interface UserAuthorizationService {

    /**
     * Check if this user has super admin access in the system
     * @param userId the internal user id (not username)
     * @return true if the user has admin access OR false otherwise
     * @throws IllegalArgumentException if the inputs are null
     */
    public boolean isUserAdmin(String userId);

    /**
     * Get a list of all user ids that have the given permission on the given resources <br/>
     * <b>Note</b>: this will be internal user ids and NOT the user EID 
     * (external user id from your provider source) 
     * 
     * @param references complete references to the resources we are checking permissions on (e.g. /collection/123)
     * @param permission a permission string (e.g. collection.create)
     * @return a Set of Strings which represent the internal user IDs of all users with that permission
     * @throws IllegalArgumentException if the inputs are null
     */
    public List<String> getUsersForPermsRefs(String[] references, String permission);

    /**
     * Get a list of all references that the given user has this permission on
     * 
     * @param userId the internal user id (not username)
     * @param permission a permission string (e.g. collection.create)
     * @return a List of reference strings which this user has the given permission on
     * @throws IllegalArgumentException if the inputs are null
     */
    public List<String> getRefsForUserPerm(String userId, String permission);

    /**
     * Get a list of all permissions that a user has on a specific reference
     * 
     * @param userId the internal user id (not username)
     * @param reference a complete reference to an object we are checking permissions on (e.g. /collection/123)
     * @return the list of all permissions that this user has on this reference
     * @throws IllegalArgumentException if the inputs are null
     */
    public List<String> getPermsForUserAndRef(String userId, String reference);

    /**
     * Check if a user has a specified permission/role for a given reference
     * 
     * @param userId the internal user id (not username)
     * @param permission a permission string (e.g. collection.create)
     * @param reference a complete reference to an object we are checking permissions on (e.g. /collection/123)
     * @return true if allowed, false otherwise
     * @throws IllegalArgumentException if the inputs are null
     */
    public boolean isUserAllowedOnRef(String userId, String permission, String reference);

}
