/**
 * $Id: UserLookupService.java 3442 2009-02-06 15:18:11Z azeckoski $
 * $URL: https://scm.dspace.org/svn/repo/dspace2/core/tags/OR09/api/src/main/java/org/dspace/services/mixins/UserLookupService.java $
 * UserLookupService.java - DSpace2 - Oct 13, 2008 11:51:15 AM - azeckoski
 **************************************************************************
 * Copyright (c) 2008 Aaron Zeckoski
 * Licensed under the Apache License, Version 2.0
 * 
 * A copy of the Apache License has been included in this 
 * distribution and is available at: http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Aaron Zeckoski (azeckoski @ gmail.com) (aaronz @ vt.edu) (aaron @ caret.cam.ac.uk)
 */

package org.dspace.services.mixins;

import java.util.List;
import java.util.Map;

import org.dspace.services.model.User;
import org.dspace.services.model.UserSearch;
import org.dspace.services.model.Evidence.UserEidType;

/**
 * Allows for looking up user information and searching for users <br/>
 * Note that all user lookups should be cached and that cache controlled by the
 * system administrator such that the developer should not be concerned about doing many
 * user lookups or many calls to the methods in this service
 * 
 * @author Aaron Zeckoski (azeckoski @ gmail.com)
 */
public interface UserLookupService {

    /**
     * Converts an eid into an internal user id,
     * if the eid is a standard one (username) then the type can be left null,
     * otherwise the type must be set to indicate the type of eid this is
     * 
     * @param userEid the user eid (cannot be null or blank)
     * @param eidType (optional) the type of eid this is
     * @return the internal user id OR null if no internal user ID can be found
     * @throws IllegalArgumentException if the userEid is not set or blank
     */
    public String getUserIdFromEid(String userEid, UserEidType eidType);

    /**
     * Translate an internal userId into a user EID
     * @param internalUserId the internal user id (not username or EID)
     * @return the user EID (often is the username) OR null if this user ID is invalid
     */
    public String getUserEIDfromID(String internalUserId);

    /**
     * Get a populated user object for the given userId
     * 
     * @param userId the internal user id (not username)
     * @return a populated {@link User} object OR null if none found
     */
    public User getUserById(String userId);

    /**
     * Get user objects populated with data based on the input set of userIds,
     * guarantees to return one {@link User} object per input id and in the
     * same order as the inputs,<br/>
     * multiple inputs version of {@link #getUserById(String)}
     * <br/>
     * Will return an empty map if null or empty array is provided
     * 
     * @param userIds an array of the internal user ids (not usernames) for users
     * @return a map of userId to {@link User} objects which match with the input ids
     */
    public Map<String, User> getUsersByIds(String[] userIds);

    /**
     * Get user objects populated with data based on a set of search parameters,
     * will return an empty list if no matches are found for this search
     * 
     * @param search the search object which defines which users should be retrieved
     * @return a list of users based on the given search
     * @throws IllegalArgumentException if the search is null
     */
    public List<User> getUsersBySearch(UserSearch search);

}
