/**
 * $Id: StorageEntity.java 3770 2009-05-12 09:43:39Z benbosman $
 * $URL: https://scm.dspace.org/svn/repo/dspace2/core/tags/OR09/api/src/main/java/org/dspace/services/model/StorageEntity.java $
 * StorageEntity.java - DS2 - Feb 2, 2009 10:37:40 AM - azeckoski
 **************************************************************************
 * Copyright (c) 2008 Aaron Zeckoski
 * Licensed under the Apache License, Version 2.0
 *
 * A copy of the Apache License has been included in this
 * distribution and is available at: http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Aaron Zeckoski (azeckoski @ gmail.com) (aaronz @ vt.edu) (aaron @ caret.cam.ac.uk)
 */

package org.dspace.services.model;

import org.dspace.services.StorageService;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Vector;

/**
 * Represents a general storage entity (file or folder),
 * when processing properties they are handled in the list order so any duplicates will be overwritten
 * as the list is processed (i.e. the last property with a given name will overwrite the first one)
 *
 * @author Aaron Zeckoski (azeckoski @ gmail.com)
 * @author Mark Diggory (mdiggory @ gmail.com)
 */
public class StorageEntity {

    public static final String ROOT_REFERENCE = StorageService.ROOT_PATH;

    protected String id;
    protected String name;
    protected String path;
    protected List<StorageProperty> properties;
    protected List<String> childrenNames;

    protected StorageEntity() {} // empty constructor for reflection usage

    /*
     * NOTE: it makes no sense to construct with children names because they have to be added later on,
     * thus children names is a read only field
     */

    /**
     * For constructing a general storage unit,
     * example reference: /trees/fruit/apple
     * @param name this is the name of this unit (e.g. apple)
     * @param path this is the path to this unit (e.g. /trees/fruit)
     */
    public StorageEntity(String name, String path) {
        this(name, path, null);
    }

    /**
     * For constructing a general storage item with properties,
     * example reference: /trees/fruit/apple
     * @param name this is the name of this unit (e.g. apple)
     * @param path this is the path to this unit (e.g. /trees/fruit)
     * @param properties the set of properties stored with this unit
     */
    public StorageEntity(String name, String path, StorageProperty[] properties) {
        super();
        if (name == null) {
            throw new IllegalArgumentException("name cannot be null and must be set");
        }
        if (path == null) {
            throw new IllegalArgumentException("path cannot be null and must be set");
        }
        this.name = name.trim();
        this.path = path.trim();
        setProperties(properties);
    }

    /**
     * @return the reference (overall unique id for this entity)
     * (various versions of this class may want to override this)
     */
    public String getReference() {
        String reference;
        if ("".equals(this.name)) {
            reference = ROOT_REFERENCE; // ROOT node
        } else if (ROOT_REFERENCE.equals(this.path)) {
            reference = "/" + this.name;
        } else {
            reference = this.path + "/" + name;
        }
        return reference;
    }

    public String getPath() {
        return path;
    }

    public void setPath(String path) {
        this.path = path;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getId() {
        return id;
    }

    public void setId(String id) {
        this.id = id;
    }

    public List<StorageProperty> getProperties() {
        return this.properties;
    }

    /**
     * Retrieve a property by name
     * @param name the name of the property (e.g. dc:title)
     * @return the storage property object if one exists OR null if none found by this name
     */
    public StorageProperty getProperty(String name) {
        StorageProperty prop = null;
        for (StorageProperty sp : properties) {
            if (name.equalsIgnoreCase(sp.getName())) {
                prop = sp;
                break;
            }
        }
        return prop;
    }

    /**
     * Retrieve all instances of a property by name
     * @param name the name of the property (e.g. dc:title)
     * @return the storage property objects found by this name, can be an empty List
     */
    public List<StorageProperty> getProperties(String name) {
        List<StorageProperty> props = new ArrayList<StorageProperty>();
        for (StorageProperty sp : properties) {
            if (name.equalsIgnoreCase(sp.getName())) {
                props.add(sp);
            }
        }
        return props;
    }

    public List<StorageRelation> getRelations(){
        List<StorageRelation> props = new ArrayList<StorageRelation>();
        for (StorageProperty sp : properties) {
            if (sp.isRelation() && sp instanceof StorageRelation) {
                props.add((StorageRelation) sp);
            }
        }
        return props;
    }

    public List<String> getChildrenNames() {
        return Collections.unmodifiableList(childrenNames);
    }

    /**
     * @return the reference to the parent of this storage entity,
     * note that {@link #ROOT_REFERENCE} ({@value #ROOT_REFERENCE}) is the ROOT node
     */
    public String getParentReference() {
        return this.path;
    }

    /**
     * For internal use only, this means you!
     * @param properties
     */
    public void setProperties(StorageProperty[] properties) {
        if (properties == null || properties.length == 0) {
            this.properties = new Vector<StorageProperty>(0);
        } else {
            this.properties = new Vector<StorageProperty>(properties.length);
            for (int i = 0; i < properties.length; i++) {
                this.properties.add(properties[i]);
            }
        }
    }

    public void addProperty(StorageProperty property) {
        if (properties == null) {
            properties = new Vector<StorageProperty>(1);
        }
        properties.add(property);
    }

    /**
     * Check whether this StorageEntity has a property by it's name
     * @param name the name of the property (e.g. dc:title)
     * @return true if this StorageEntity has a property by that name, false otherwise
     */
    public boolean hasProperty(String name){
        return getProperty(name) != null;
    }

    /**
     * For internal use only, this means you!
     * @param childrenNames
     */
    public void setChildrenNames(String[] childrenNames) {
        if (childrenNames == null || childrenNames.length == 0) {
            this.childrenNames = new Vector<String>(0);
        } else {
            this.childrenNames = new Vector<String>(childrenNames.length);
            for (int i = 0; i < childrenNames.length; i++) {
                String name = childrenNames[i];
                if (! this.childrenNames.contains(name)) {
                    this.childrenNames.add(name);
                }
            }
        }
    }

    @Override
    public boolean equals(Object obj) {
        if (null == obj) {
            return false;
        }
        if (!(obj instanceof StorageEntity)) {
            return false;
        } else {
            StorageEntity castObj = (StorageEntity) obj;
            boolean eq = (this.name == null ? castObj.name == null : this.name.equals(castObj.name))
            && (this.path == null ? castObj.path == null : this.path.equals(castObj.path));
            return eq;
        }
    }

    @Override
    public int hashCode() {
        String hashStr = this.getClass().getName() + ":" + this.name + ":" + this.path;
        return hashStr.hashCode();
    }

    @Override
    public String toString() {
        return "entity::ref=" + getReference() + ":id=" + this.id + ":name=" + this.name + ":path="
        + this.path;
    }

}
