/**
 * $Id: User.java 3678 2009-04-06 12:45:44Z grahamtriggs $
 * $URL: https://scm.dspace.org/svn/repo/dspace2/core/tags/OR09/api/src/main/java/org/dspace/services/model/User.java $
 * User.java - DSpace2 - Oct 13, 2008 11:53:13 AM - azeckoski
 **************************************************************************
 * Copyright (c) 2008 Aaron Zeckoski
 * Licensed under the Apache License, Version 2.0
 * 
 * A copy of the Apache License has been included in this 
 * distribution and is available at: http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Aaron Zeckoski (azeckoski @ gmail.com) (aaronz @ vt.edu) (aaron @ caret.cam.ac.uk)
 */

package org.dspace.services.model;

import java.io.Serializable;
import java.util.Date;
import java.util.Map;

/**
 * This represents a user in the system
 * 
 * @author Aaron Zeckoski (azeckoski @ gmail.com)
 */
public class User implements Serializable {
	private static final long serialVersionUID = 1L;

    public static enum UserType {INTERNAL, EXTERNAL};

    /**
     * Indicates the fields which are persisted
     */
    public static final String[] persistentFields = {
        "id", "eid", "password", "email", "displayName", "sortName", "ownerId", "createdOn", "lastModified"
    };

    private String id;
    private String eid;
    private String password;
    private String email;
    private String displayName;
    private String sortName;
    private UserType type = UserType.INTERNAL;
    private String ownerId;
    private Date createdOn;
    private Date lastModified;
    public Map<String, String> props;

    /**
     * Recommend you use one of the other constructors, this is for internal usage
     */
    public User() {} // empty constructor for internal usage

    public User(String eid, String email, String ownerId) {
        this(eid, email, ownerId, null, null, null);
    }

    public User(String eid, String email, String ownerId, String displayName, String sortName) {
        this(eid, email, ownerId, displayName, sortName, null);
    }

    public User(String eid, String email, String ownerId, String displayName, String sortName, String password) {
        if (eid == null) {
            throw new IllegalArgumentException("eid cannot be null and must be set");
        }
        if (ownerId == null) {
            throw new IllegalArgumentException("ownerId cannot be null and must be set");
        }
        this.eid = eid;
        this.ownerId = ownerId;
        this.email = email;
        this.password = password;
        if (displayName == null) {
            this.displayName = this.email == null ? this.eid : this.email;
        } else {
            this.displayName = displayName;
        }
        if (sortName == null) {
            this.sortName = this.eid;
        } else {
            this.sortName = sortName;
        }
        this.lastModified = new Date();
        this.createdOn = this.lastModified;
    }


    public void setLastModified(Date lastModified) {
        this.lastModified = new Date(lastModified.getTime());
    }

    public Date getCreatedOn() {
        return new Date(createdOn.getTime());
    }

    public void setCreatedOn(Date createdOn) {
        this.createdOn = new Date(createdOn.getTime());
    }

    public String getId() {
        return id;
    }

    public void setId(String id) {
        this.id = id;
    }

    public String getEid() {
        return eid;
    }

    public void setEid(String eid) {
        this.eid = eid;
    }

    public String getPassword() {
        return password;
    }

    public void setPassword(String password) {
        this.password = password;
    }

    public String getEmail() {
        return email;
    }

    public void setEmail(String email) {
        this.email = email;
    }

    public String getSortName() {
        return sortName;
    }

    public void setSortName(String sortName) {
        this.sortName = sortName;
    }

    public String getDisplayName() {
        return displayName;
    }

    public void setDisplayName(String displayName) {
        this.displayName = displayName;
    }

    public UserType getType() {
        return type;
    }

    public void setType(UserType type) {
        this.type = type;
    }

    public String getOwnerId() {
        return ownerId;
    }

    public void setOwnerId(String owner) {
        this.ownerId = owner;
    }

    public Map<String, String> getProps() {
        return props;
    }

    public void setProps(Map<String, String> props) {
        this.props = props;
    }

    @Override
    public boolean equals(Object obj) {
        if (null == obj) {
            return false;
        }
        if (!(obj instanceof User)) {
            return false;
        } else {
            User castObj = (User) obj;
            boolean eq = (this.id == null ? castObj.id == null : this.id.equals(castObj.id))
            && (this.eid == null ? castObj.eid == null : this.eid.equals(castObj.eid));
            return eq;
        }
    }

    @Override
    public int hashCode() {
        String hashStr = this.getClass().getName() + ":" + this.id + ":" + this.eid;
        return hashStr.hashCode();
    }

    @Override
    public String toString() {
        return "user::id="+this.id+":eid="+this.eid+":email="+this.email+":type="+this.type;
    }

}
